import { useState, useEffect } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';
import PageHeader from '../../components/PageHeader';

/**
 * 🎨 MÓDULO: HOSTING - LISTADO DE SERVICIOS
 * 
 * ✅ ESTILO ACTUALIZADO - Usa el diseño premium estándar
 */

interface HostingService {
    id: number;
    label: string;
    domain_fqdn?: string;
    status: string;
    next_due_at: string;
    price: number;
    currency: string;
    auto_renew: boolean;
    plan?: { name: string };
    server?: { name: string };
}

const statusConfig: Record<string, { label: string; class: string; emoji: string }> = {
    active: { label: 'Activo', class: 'badge-success', emoji: '✅' },
    suspended: { label: 'Suspendido', class: 'badge-warning', emoji: '⏸️' },
    cancelled: { label: 'Cancelado', class: 'badge-danger', emoji: '❌' },
    pending_setup: { label: 'Pendiente', class: 'badge-default', emoji: '⏳' },
    terminated: { label: 'Terminado', class: 'badge-default', emoji: '🔚' },
};

export default function HostingIndex() {
    const [services, setServices] = useState<HostingService[]>([]);
    const [loading, setLoading] = useState(true);
    const [filter, setFilter] = useState<'all' | 'expiring'>('all');
    const [showModal, setShowModal] = useState(false);
    const [plans, setPlans] = useState<any[]>([]);
    const [servers, setServers] = useState<any[]>([]);
    const [formData, setFormData] = useState({
        label: '',
        domain_fqdn: '',
        plan_id: '',
        server_id: '',
        status: 'active',
        started_at: new Date().toISOString().split('T')[0],
        next_due_at: '',
        billing_cycle: 'annual',
        price: '',
        currency: 'USD',
    });

    useEffect(() => {
        fetchServices();
        fetchPlans();
        fetchServers();
    }, [filter]);

    const fetchServices = async () => {
        try {
            const params = filter === 'expiring' ? { expiring_days: 30 } : {};
            const res = await axios.get('/api/hosting/services', { params });
            setServices(res.data);
        } catch (error) {
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    const fetchPlans = async () => {
        try {
            const res = await axios.get('/api/hosting/plans');
            setPlans(res.data);
        } catch (error) {
            console.error(error);
        }
    };

    const fetchServers = async () => {
        try {
            const res = await axios.get('/api/hosting/servers');
            setServers(res.data);
        } catch (error) {
            console.error(error);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        try {
            await axios.post('/api/hosting/services', {
                ...formData,
                plan_id: formData.plan_id ? parseInt(formData.plan_id) : null,
                server_id: formData.server_id ? parseInt(formData.server_id) : null,
                price: parseFloat(formData.price),
            });
            alert('Servicio creado correctamente');
            setShowModal(false);
            fetchServices();
        } catch (error) {
            alert('Error al crear servicio');
        }
    };

    const getDaysUntilExpiration = (expiresAt: string) => {
        return Math.ceil((new Date(expiresAt).getTime() - Date.now()) / (1000 * 60 * 60 * 24));
    };

    const stats = [
        { icon: '🖥️', label: 'Total Servicios', value: services.length },
        { icon: '✅', label: 'Activos', value: services.filter(s => s.status === 'active').length },
        { icon: '⚠️', label: 'Vencen <30 días', value: services.filter(s => getDaysUntilExpiration(s.next_due_at) <= 30 && getDaysUntilExpiration(s.next_due_at) > 0).length },
    ];

    if (loading) return <div className="p-8 text-center text-slate-400">Cargando servicios...</div>;

    return (
        <div className="animate-fadeIn">
            <PageHeader
                title="Servicios de Hosting"
                subtitle="Gestión de suscripciones y servidores"
                actions={
                    <>
                        <button onClick={() => setShowModal(true)} className="btn btn-primary mr-2">
                            ➕ Nuevo Servicio
                        </button>
                        <Link to="/app/hosting/plans" className="btn btn-secondary">📋 Planes</Link>
                    </>
                }
                stats={stats}
            />

            {/* Filtros */}
            <div className="flex gap-2 mb-4">
                <button
                    className={`btn ${filter === 'all' ? 'btn-primary' : 'btn-ghost'}`}
                    onClick={() => setFilter('all')}
                >
                    Todos
                </button>
                <button
                    className={`btn ${filter === 'expiring' ? 'btn-warning' : 'btn-ghost'}`}
                    onClick={() => setFilter('expiring')}
                >
                    ⚠️ Próximos a Vencer
                </button>
            </div>

            <div className="table-container">
                <div className="table-header">
                    <h2 className="table-title">🖥️ Servicios Activos</h2>
                </div>
                <div className="table-wrapper">
                    <table className="table">
                        <thead>
                            <tr>
                                <th>Servicio</th>
                                <th>Plan</th>
                                <th>Estado</th>
                                <th>Vence</th>
                                <th>Precio</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            {services.length === 0 ? (
                                <tr>
                                    <td colSpan={6} className="text-center p-8 text-slate-500">
                                        No hay servicios registrados
                                    </td>
                                </tr>
                            ) : (
                                services.map((service) => {
                                    const daysLeft = getDaysUntilExpiration(service.next_due_at);
                                    const isExpiring = daysLeft <= 30 && daysLeft > 0;

                                    return (
                                        <tr key={service.id} className={isExpiring ? 'bg-orange-500/10' : ''}>
                                            <td>
                                                <div className="font-bold text-white">{service.label}</div>
                                                {service.domain_fqdn && <div className="text-xs text-slate-400">{service.domain_fqdn}</div>}
                                            </td>
                                            <td className="text-sm">{service.plan?.name || 'N/A'}</td>
                                            <td>
                                                <span className={`badge ${statusConfig[service.status]?.class || 'badge-default'}`}>
                                                    {statusConfig[service.status]?.emoji} {statusConfig[service.status]?.label}
                                                </span>
                                            </td>
                                            <td>
                                                <div className={daysLeft <= 7 ? 'text-red-400 font-bold' : ''}>
                                                    {service.next_due_at}
                                                </div>
                                                {isExpiring && (
                                                    <div className="text-xs text-orange-400">⚠️ {daysLeft} días</div>
                                                )}
                                            </td>
                                            <td className="font-mono">{service.currency} {service.price}</td>
                                            <td>
                                                <Link to={`/app/hosting/services/${service.id}`} className="btn btn-sm btn-ghost hover:text-blue-400">
                                                    🔧 Gestionar
                                                </Link>
                                            </td>
                                        </tr>
                                    );
                                })
                            )}
                        </tbody>
                    </table>
                </div>
            </div>

            {/* Modal Crear Servicio - Estilo Marketing */}
            {showModal && (
                <div style={{
                    position: 'fixed',
                    inset: 0,
                    background: 'rgba(0, 0, 0, 0.7)',
                    backdropFilter: 'blur(4px)',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    zIndex: 9999,
                    padding: '1rem'
                }} onClick={() => setShowModal(false)}>
                    <div className="card" style={{
                        maxWidth: '600px',
                        width: '100%',
                        maxHeight: '90vh',
                        overflowY: 'auto'
                    }} onClick={(e) => e.stopPropagation()}>
                        <div style={{ padding: '1.5rem', borderBottom: '1px solid var(--border-color)' }}>
                            <h2 style={{ fontSize: '1.5rem', fontWeight: 700, color: 'var(--text-primary)' }}>
                                🖥️ Nuevo Servicio de Hosting
                            </h2>
                            <p style={{ fontSize: '0.9rem', color: 'var(--text-muted)', marginTop: '0.5rem' }}>
                                Crea un nuevo servicio de hosting para un cliente
                            </p>
                        </div>
                        <form onSubmit={handleSubmit} style={{ padding: '1.5rem' }}>
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '1.25rem' }}>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Nombre del servicio
                                    </label>
                                    <input
                                        type="text"
                                        required
                                        className="input"
                                        placeholder="Ej: Hosting Cliente XYZ"
                                        value={formData.label}
                                        onChange={(e) => setFormData({ ...formData, label: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Dominio (FQDN)
                                    </label>
                                    <input
                                        type="text"
                                        className="input"
                                        placeholder="ejemplo.com"
                                        value={formData.domain_fqdn}
                                        onChange={(e) => setFormData({ ...formData, domain_fqdn: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Plan
                                    </label>
                                    <select
                                        className="input"
                                        value={formData.plan_id}
                                        onChange={(e) => setFormData({ ...formData, plan_id: e.target.value })}
                                    >
                                        <option value="">Seleccionar plan</option>
                                        {plans.map(plan => (
                                            <option key={plan.id} value={plan.id}>{plan.name}</option>
                                        ))}
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Servidor
                                    </label>
                                    <select
                                        className="input"
                                        value={formData.server_id}
                                        onChange={(e) => setFormData({ ...formData, server_id: e.target.value })}
                                    >
                                        <option value="">Seleccionar servidor</option>
                                        {servers.map(server => (
                                            <option key={server.id} value={server.id}>{server.name}</option>
                                        ))}
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Ciclo de facturación
                                    </label>
                                    <select
                                        className="input"
                                        value={formData.billing_cycle}
                                        onChange={(e) => setFormData({ ...formData, billing_cycle: e.target.value })}
                                    >
                                        <option value="monthly">Mensual</option>
                                        <option value="quarterly">Trimestral</option>
                                        <option value="semiannual">Semestral</option>
                                        <option value="annual">Anual</option>
                                        <option value="biennial">Bienal</option>
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Precio
                                    </label>
                                    <input
                                        type="number"
                                        step="0.01"
                                        required
                                        className="input"
                                        placeholder="99.99"
                                        value={formData.price}
                                        onChange={(e) => setFormData({ ...formData, price: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Fecha de inicio
                                    </label>
                                    <input
                                        type="date"
                                        required
                                        className="input"
                                        value={formData.started_at}
                                        onChange={(e) => setFormData({ ...formData, started_at: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Próximo vencimiento
                                    </label>
                                    <input
                                        type="date"
                                        required
                                        className="input"
                                        value={formData.next_due_at}
                                        onChange={(e) => setFormData({ ...formData, next_due_at: e.target.value })}
                                    />
                                </div>
                                <div style={{ display: 'flex', gap: '0.75rem', justifyContent: 'flex-end', marginTop: '1rem' }}>
                                    <button type="button" className="btn btn-secondary" onClick={() => setShowModal(false)}>
                                        Cancelar
                                    </button>
                                    <button type="submit" className="btn btn-primary">
                                        💾 Guardar Servicio
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
