import { useState, useEffect } from 'react';
import axios from 'axios';
import PageHeader from '../../components/PageHeader';

/**
 * 🎨 MÓDULO: HOSTING - SERVIDORES
 * 
 * ✅ ESTILO ACTUALIZADO - CRUD Completo
 */

export default function HostingServers() {
    const [servers, setServers] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);
    const [showModal, setShowModal] = useState(false);
    const [formData, setFormData] = useState({
        name: '',
        provider: 'hetzner',
        server_type: 'vps',
        region: '',
        ip_primary: '',
        ssh_port: '22',
        notes: '',
        monitoring_enabled: false,
    });

    useEffect(() => {
        fetchServers();
    }, []);

    const fetchServers = async () => {
        try {
            const res = await axios.get('/api/hosting/servers');
            setServers(res.data);
        } catch (error) {
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        try {
            await axios.post('/api/hosting/servers', {
                ...formData,
                ssh_port: parseInt(formData.ssh_port),
            });
            alert('Servidor creado correctamente');
            setShowModal(false);
            fetchServers();

            // Reset form
            setFormData({
                name: '',
                provider: 'hetzner',
                server_type: 'vps',
                region: '',
                ip_primary: '',
                ssh_port: '22',
                notes: '',
                monitoring_enabled: false,
            });
        } catch (error) {
            alert('Error al crear servidor');
        }
    };

    if (loading) return <div className="p-8 text-center">Cargando servidores...</div>;

    return (
        <div className="animate-fadeIn">
            <PageHeader
                title="Servidores"
                subtitle="Infraestructura de hosting"
                actions={
                    <button onClick={() => setShowModal(true)} className="btn btn-primary">
                        ➕ Nuevo Servidor
                    </button>
                }
            />

            <div className="table-container">
                <div className="table-wrapper">
                    <table className="table">
                        <thead>
                            <tr>
                                <th>Nombre</th>
                                <th>Proveedor</th>
                                <th>Tipo</th>
                                <th>IP</th>
                                <th>Servicios</th>
                                <th>Estado</th>
                            </tr>
                        </thead>
                        <tbody>
                            {servers.length === 0 ? (
                                <tr>
                                    <td colSpan={6} className="text-center p-8 text-slate-500">
                                        No hay servidores registrados
                                    </td>
                                </tr>
                            ) : (
                                servers.map((server) => (
                                    <tr key={server.id}>
                                        <td className="font-bold">{server.name}</td>
                                        <td><span className="badge badge-primary">{server.provider}</span></td>
                                        <td>{server.server_type}</td>
                                        <td className="font-mono text-sm">{server.ip_primary}</td>
                                        <td>{server.services_count || 0}</td>
                                        <td>
                                            <span className={`badge ${server.monitoring_enabled ? 'badge-success' : 'badge-default'}`}>
                                                {server.monitoring_enabled ? '✓ Monitoreado' : 'Sin monitoreo'}
                                            </span>
                                        </td>
                                    </tr>
                                ))
                            )}
                        </tbody>
                    </table>
                </div>
            </div>

            {/* Modal Crear Servidor - Estilo Marketing */}
            {showModal && (
                <div style={{
                    position: 'fixed',
                    inset: 0,
                    background: 'rgba(0, 0, 0, 0.7)',
                    backdropFilter: 'blur(4px)',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    zIndex: 9999,
                    padding: '1rem'
                }} onClick={() => setShowModal(false)}>
                    <div className="card" style={{
                        maxWidth: '600px',
                        width: '100%',
                        maxHeight: '90vh',
                        overflowY: 'auto'
                    }} onClick={(e) => e.stopPropagation()}>
                        <div style={{ padding: '1.5rem', borderBottom: '1px solid var(--border-color)' }}>
                            <h2 style={{ fontSize: '1.5rem', fontWeight: 700, color: 'var(--text-primary)' }}>
                                🖥️ Nuevo Servidor
                            </h2>
                            <p style={{ fontSize: '0.9rem', color: 'var(--text-muted)', marginTop: '0.5rem' }}>
                                Agrega un nuevo servidor a tu infraestructura
                            </p>
                        </div>
                        <form onSubmit={handleSubmit} style={{ padding: '1.5rem' }}>
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '1.25rem' }}>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Nombre del servidor
                                    </label>
                                    <input
                                        type="text"
                                        required
                                        className="input"
                                        placeholder="Ej: Server-01"
                                        value={formData.name}
                                        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Proveedor
                                    </label>
                                    <select
                                        className="input"
                                        value={formData.provider}
                                        onChange={(e) => setFormData({ ...formData, provider: e.target.value })}
                                    >
                                        <option value="hetzner">Hetzner</option>
                                        <option value="ovh">OVH</option>
                                        <option value="digitalocean">DigitalOcean</option>
                                        <option value="aws">AWS</option>
                                        <option value="vultr">Vultr</option>
                                        <option value="local">Local</option>
                                        <option value="other">Otro</option>
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Tipo de servidor
                                    </label>
                                    <select
                                        className="input"
                                        value={formData.server_type}
                                        onChange={(e) => setFormData({ ...formData, server_type: e.target.value })}
                                    >
                                        <option value="vps">VPS</option>
                                        <option value="dedicated">Dedicado</option>
                                        <option value="node">Nodo</option>
                                        <option value="shared_cluster">Cluster Compartido</option>
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Región
                                    </label>
                                    <input
                                        type="text"
                                        className="input"
                                        placeholder="Ej: EU-Central"
                                        value={formData.region}
                                        onChange={(e) => setFormData({ ...formData, region: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        IP principal
                                    </label>
                                    <input
                                        type="text"
                                        required
                                        className="input"
                                        placeholder="192.168.1.1"
                                        value={formData.ip_primary}
                                        onChange={(e) => setFormData({ ...formData, ip_primary: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Puerto SSH
                                    </label>
                                    <input
                                        type="number"
                                        className="input"
                                        placeholder="22"
                                        value={formData.ssh_port}
                                        onChange={(e) => setFormData({ ...formData, ssh_port: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Notas
                                    </label>
                                    <textarea
                                        className="input"
                                        rows={3}
                                        placeholder="Información adicional..."
                                        value={formData.notes}
                                        onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                                    />
                                </div>
                                <div style={{ display: 'flex', alignItems: 'center', gap: '0.5rem' }}>
                                    <input
                                        type="checkbox"
                                        id="monitoring"
                                        checked={formData.monitoring_enabled}
                                        onChange={(e) => setFormData({ ...formData, monitoring_enabled: e.target.checked })}
                                    />
                                    <label htmlFor="monitoring" style={{ fontSize: '0.875rem' }}>Habilitar monitoreo</label>
                                </div>
                                <div style={{ display: 'flex', gap: '0.75rem', justifyContent: 'flex-end', marginTop: '1rem' }}>
                                    <button type="button" className="btn btn-secondary" onClick={() => setShowModal(false)}>
                                        Cancelar
                                    </button>
                                    <button type="submit" className="btn btn-primary">
                                        💾 Guardar Servidor
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
