import React, { useState } from 'react';
import axios from 'axios';
import { useTheme } from '../context/ThemeContext';

export default function Login() {
    const [email, setEmail] = useState('');
    const [password, setPassword] = useState('');
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState('');
    const { theme, toggleTheme } = useTheme();

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setLoading(true);
        setError('');

        try {
            // Get CSRF cookie first for Sanctum
            await axios.get('/sanctum/csrf-cookie');

            await axios.post('/api/login', {
                email,
                password,
            });

            // Redirect on success
            window.location.href = '/app/dashboard';
        } catch (err: any) {
            setLoading(false);
            if (err.response) {
                // Server responded with a status code
                const status = err.response.status;
                const message = err.response.data.message || err.response.statusText;

                if (status === 422) {
                    setError(message);
                } else if (status === 419) {
                    setError('Error de seguridad (CSRF). Recarga la página.');
                } else {
                    setError(`Error del servidor (${status}): ${message}`);
                }
            } else if (err.request) {
                // Request made but no response
                setError('No hay respuesta del servidor. Revisa tu conexión.');
            } else {
                setError(`Error: ${err.message}`);
            }
            console.error('Login Error:', err);
        }
    };

    return (
        <div style={{
            minHeight: '100vh',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            padding: '1rem',
            position: 'relative',
            overflow: 'hidden',
        }}>
            {/* Background Effects */}
            <div className="login-background" style={{
                position: 'absolute',
                inset: 0,
                zIndex: 0,
                background: 'var(--bg-primary)',
                transition: 'background 0.5s ease',
            }} />

            {/* Gradient Overlay using CSS variables */}
            <div style={{
                position: 'absolute',
                inset: 0,
                zIndex: 0,
                background: `
                    radial-gradient(ellipse at 20% 20%, var(--color-primary-glow) 0%, transparent 50%),
                    radial-gradient(ellipse at 80% 80%, var(--color-accent-glow) 0%, transparent 50%),
                    radial-gradient(ellipse at 50% 50%, var(--color-secondary-glow) 0%, transparent 50%)
                `,
                opacity: 0.4
            }} />

            {/* Floating orbs */}
            <div style={{
                position: 'absolute',
                top: '20%',
                left: '10%',
                width: '300px',
                height: '300px',
                background: 'radial-gradient(circle, var(--color-primary-glow) 0%, transparent 70%)',
                borderRadius: '50%',
                filter: 'blur(40px)',
                animation: 'float 6s ease-in-out infinite',
            }} />
            <div style={{
                position: 'absolute',
                bottom: '20%',
                right: '10%',
                width: '250px',
                height: '250px',
                background: 'radial-gradient(circle, var(--color-accent-glow) 0%, transparent 70%)',
                borderRadius: '50%',
                filter: 'blur(40px)',
                animation: 'float 8s ease-in-out infinite reverse',
            }} />

            {/* Theme Toggle */}
            <button
                onClick={toggleTheme}
                className="theme-toggle-btn"
                style={{
                    position: 'absolute',
                    top: '1.5rem',
                    right: '1.5rem',
                    zIndex: 50,
                    width: '48px',
                    height: '48px',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    background: 'var(--bg-card)',
                    border: '1px solid var(--border-color)',
                    color: 'var(--text-primary)',
                    borderRadius: '12px',
                    cursor: 'pointer',
                    fontSize: '1.5rem',
                    transition: 'all 0.3s ease',
                    boxShadow: 'var(--shadow-md)',
                }}
                title={theme === 'dark' ? 'Cambiar a modo claro' : 'Cambiar a modo oscuro'}
            >
                {theme === 'dark' ? '☀️' : '🌙'}
            </button>

            {/* Login Card */}
            <div
                className="card animate-slideUp"
                style={{
                    width: '100%',
                    maxWidth: '420px',
                    position: 'relative',
                    zIndex: 1,
                    padding: '2.5rem',
                }}
            >
                {/* Logo Area */}
                <div style={{ textAlign: 'center', marginBottom: '2rem' }}>
                    <div style={{
                        width: '72px',
                        height: '72px',
                        margin: '0 auto 1rem',
                        borderRadius: '18px',
                        background: 'linear-gradient(135deg, var(--color-primary), var(--color-accent))',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        boxShadow: '0 8px 32px hsla(217, 91%, 60%, 0.3)',
                    }}>
                        {/* Placeholder for logo - replace with <img src="/logo.svg" /> */}
                        <span style={{ fontSize: '2rem', color: 'white' }}>🚀</span>
                    </div>
                    <h1 style={{
                        fontSize: '1.75rem',
                        fontWeight: 800,
                        color: 'var(--text-primary)',
                        marginBottom: '0.25rem',
                        letterSpacing: '-0.02em',
                    }}>
                        Cibertronia
                    </h1>
                    <p style={{
                        color: 'var(--text-muted)',
                        fontSize: '0.9rem',
                        fontWeight: 500,
                    }}>
                        Service Hub • Panel de Administración
                    </p>
                </div>

                {/* Error Message */}
                {error && (
                    <div style={{
                        padding: '0.875rem 1rem',
                        background: 'rgba(239, 68, 68, 0.1)',
                        border: '1px solid rgba(239, 68, 68, 0.2)',
                        borderRadius: '10px',
                        color: 'var(--color-danger)',
                        fontSize: '0.85rem',
                        marginBottom: '1.25rem',
                        textAlign: 'center',
                    }}>
                        {error}
                    </div>
                )}

                <form onSubmit={handleSubmit}>
                    <div className="form-group">
                        <label className="form-label">Email</label>
                        <input
                            type="email"
                            className="form-input"
                            placeholder="tu@email.com"
                            value={email}
                            onChange={(e) => setEmail(e.target.value)}
                            required
                            autoComplete="email"
                            autoFocus
                        />
                    </div>

                    <div className="form-group">
                        <label className="form-label">Contraseña</label>
                        <input
                            type="password"
                            className="form-input"
                            placeholder="••••••••"
                            value={password}
                            onChange={(e) => setPassword(e.target.value)}
                            required
                            autoComplete="current-password"
                        />
                    </div>

                    <div style={{
                        display: 'flex',
                        justifyContent: 'space-between',
                        alignItems: 'center',
                        marginBottom: '1.5rem',
                        fontSize: '0.85rem',
                    }}>
                        <label style={{
                            display: 'flex',
                            alignItems: 'center',
                            gap: '0.5rem',
                            cursor: 'pointer',
                            color: 'var(--text-secondary)',
                        }}>
                            <input
                                type="checkbox"
                                style={{
                                    width: '16px',
                                    height: '16px',
                                    accentColor: 'var(--color-primary)',
                                }}
                            />
                            Recordarme
                        </label>
                        <a
                            href="/forgot-password"
                            style={{
                                color: 'var(--color-primary)',
                                textDecoration: 'none',
                                fontWeight: 500,
                            }}
                        >
                            ¿Olvidaste tu contraseña?
                        </a>
                    </div>

                    <button
                        type="submit"
                        className="btn btn-primary btn-lg"
                        style={{ width: '100%' }}
                        disabled={loading}
                    >
                        {loading ? (
                            <>
                                <span style={{
                                    display: 'inline-block',
                                    width: '16px',
                                    height: '16px',
                                    border: '2px solid white',
                                    borderTopColor: 'transparent',
                                    borderRadius: '50%',
                                    animation: 'spin 1s linear infinite',
                                }} />
                                Iniciando sesión...
                            </>
                        ) : (
                            '🔐 Iniciar sesión'
                        )}
                    </button>
                </form>

                {/* Demo Credentials */}
                <div style={{
                    marginTop: '1.5rem',
                    padding: '1rem',
                    background: 'var(--glass-bg)',
                    borderRadius: '10px',
                    fontSize: '0.8rem',
                    textAlign: 'center',
                    color: 'var(--text-muted)',
                }}>
                    <div style={{ fontWeight: 600, marginBottom: '0.5rem', color: 'var(--text-secondary)' }}>
                        🔑 Credenciales de demo
                    </div>
                    <div>Email: <code style={{ color: 'var(--color-primary)' }}>admin@cibertronia.com</code></div>
                    <div>Password: <code style={{ color: 'var(--color-primary)' }}>password</code></div>
                </div>

                <div style={{
                    textAlign: 'center',
                    marginTop: '1.5rem',
                    paddingTop: '1.5rem',
                    borderTop: '1px solid var(--border-color)',
                    fontSize: '0.85rem',
                    color: 'var(--text-muted)',
                }}>
                    ¿No tienes cuenta?{' '}
                    <a
                        href="/register"
                        style={{
                            color: 'var(--color-primary)',
                            textDecoration: 'none',
                            fontWeight: 600,
                        }}
                    >
                        Regístrate aquí
                    </a>
                </div>
            </div>

            {/* Footer */}
            <div style={{
                position: 'absolute',
                bottom: '1.5rem',
                left: '50%',
                transform: 'translateX(-50%)',
                fontSize: '0.75rem',
                color: 'var(--text-muted)',
                zIndex: 1,
            }}>
                © 2026 Cibertronia Service. Todos los derechos reservados.
            </div>

            {/* CSS Animations */}
            <style>{`
                @keyframes float {
                    0%, 100% { transform: translateY(0px); }
                    50% { transform: translateY(-20px); }
                }
                @keyframes spin {
                    from { transform: rotate(0deg); }
                    to { transform: rotate(360deg); }
                }
            `}</style>
        </div>
    );
}
