import { useState, useEffect, CSSProperties } from 'react';
import axios from 'axios';
import PageHeader from '../../components/PageHeader';

interface Client {
    id: number;
    name: string;
}

interface Campaign {
    id: number;
    name: string;
    budget: number;
    start_date: string;
    status: 'planning' | 'active' | 'completed';
    platform: string | string[];
    client?: Client;
}

/**
 * 📣 PÁGINA: AGENCIA DE MARKETING
 * 
 * ✅ ESTILO ACTUALIZADO - Usa el mismo diseño premium del Dashboard
 * 
 * Componentes usados:
 * - PageHeader: Header estándar con título, subtítulo y acciones
 * - Metric Cards: Tarjetas de métricas con gradientes e iconos
 * - Table Container: Tabla con el estilo glassmorphism estándar
 */
export default function MarketingIndex() {
    const [campaigns, setCampaigns] = useState<Campaign[]>([]);
    const [loading, setLoading] = useState(true);
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [clients, setClients] = useState<Client[]>([]);

    // Form state
    const [formData, setFormData] = useState({
        client_id: '',
        name: '',
        budget: '',
        start_date: '',
        status: 'planning' as 'planning' | 'active' | 'completed',
        platform: [] as string[]
    });

    useEffect(() => {
        fetchData();
    }, []);

    const fetchData = async () => {
        setLoading(true);
        try {
            const [campaignsRes, clientsRes] = await Promise.all([
                axios.get('/api/marketing-campaigns'),
                axios.get('/api/clients').catch(() => ({ data: [] }))
            ]);

            setCampaigns(Array.isArray(campaignsRes.data) ? campaignsRes.data : []);
            setClients(Array.isArray(clientsRes.data) ? clientsRes.data : []);
        } catch (error) {
            console.error('Error fetching data:', error);
        } finally {
            setLoading(false);
        }
    };

    const handleCreate = async (e: React.FormEvent) => {
        e.preventDefault();
        try {
            await axios.post('/api/marketing-campaigns', formData);
            setIsModalOpen(false);
            fetchData();
            setFormData({
                client_id: '',
                name: '',
                budget: '',
                start_date: '',
                status: 'planning',
                platform: []
            });
        } catch (error) {
            console.error('Error creating campaign:', error);
        }
    };

    const getStatusConfig = (status: string) => {
        switch (status) {
            case 'active':
                return { label: 'Activa', class: 'badge-success' };
            case 'completed':
                return { label: 'Completada', class: 'badge-primary' };
            default:
                return { label: 'Planificación', class: 'badge-warning' };
        }
    };

    // Calculate metrics
    const activeCampaigns = campaigns.filter(c => c.status === 'active').length;
    const completedCampaigns = campaigns.filter(c => c.status === 'completed').length;
    const totalBudget = campaigns.reduce((acc, c) => acc + Number(c.budget), 0);

    const metricCards = [
        {
            icon: '🚀',
            label: 'Campañas Activas',
            value: activeCampaigns.toString(),
            subtext: `${campaigns.length} totales`,
            gradient: 'linear-gradient(135deg, #10b981, #059669)',
            glow: 'rgba(16, 185, 129, 0.4)',
        },
        {
            icon: '💰',
            label: 'Presupuesto Total',
            value: `$${totalBudget.toLocaleString('es-ES', { minimumFractionDigits: 2 })}`,
            gradient: 'linear-gradient(135deg, #3b82f6, #2563eb)',
            glow: 'rgba(59, 130, 246, 0.4)',
        },
        {
            icon: '✅',
            label: 'Completadas',
            value: completedCampaigns.toString(),
            gradient: 'linear-gradient(135deg, #8b5cf6, #7c3aed)',
            glow: 'rgba(139, 92, 246, 0.4)',
        },
    ];

    if (loading) {
        return (
            <div className="animate-fadeIn" style={{ display: 'flex', alignItems: 'center', justifyContent: 'center', minHeight: '400px' }}>
                <div style={{ textAlign: 'center', color: 'var(--text-muted)' }}>
                    <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>⏳</div>
                    <div>Cargando campañas...</div>
                </div>
            </div>
        );
    }

    return (
        <div className="animate-fadeIn">
            {/* Page Header - Componente estándar */}
            <PageHeader
                title="Agencia de Marketing"
                subtitle="Gestiona tus campañas publicitarias y creatividades"
                actions={
                    <>
                        <a href="/api/marketing-campaigns/export?type=xlsx" className="btn btn-secondary">
                            📊 Excel
                        </a>
                        <a href="/api/marketing-campaigns/export?type=pdf" className="btn btn-secondary">
                            📄 PDF
                        </a>
                        <button className="btn btn-primary" onClick={() => setIsModalOpen(true)}>
                            ✨ Nueva Campaña
                        </button>
                    </>
                }
                stats={[
                    { icon: '📅', label: 'Hoy', value: new Date().toLocaleDateString('es-ES', { weekday: 'long', day: 'numeric', month: 'short' }) },
                    { icon: '📊', label: 'Total campañas', value: campaigns.length },
                ]}
            />

            {/* Metric Cards Grid - Mismo estilo del Dashboard */}
            <div className="cards-grid">
                {metricCards.map((card, index) => (
                    <div
                        key={index}
                        className="card metric-card"
                        style={{ '--card-glow': card.glow } as CSSProperties}
                    >
                        <div
                            className="metric-icon"
                            style={{ background: card.gradient }}
                        >
                            {card.icon}
                        </div>
                        <div className="metric-content">
                            <div className="metric-label">{card.label}</div>
                            <div className="metric-value">{card.value}</div>
                            {card.subtext && (
                                <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)', marginTop: '0.25rem' }}>
                                    {card.subtext}
                                </div>
                            )}
                        </div>
                    </div>
                ))}
            </div>

            {/* Campaigns Table - Mismo estilo del Dashboard */}
            <div style={{ marginTop: '2.5rem' }}>
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">📣 Campañas de Marketing</h2>
                        <button className="btn btn-ghost btn-sm" onClick={fetchData}>
                            🔄 Actualizar
                        </button>
                    </div>
                    <div className="table-wrapper">
                        <table className="table">
                            <thead>
                                <tr>
                                    <th>Campaña</th>
                                    <th>Cliente</th>
                                    <th>Presupuesto</th>
                                    <th>Estado</th>
                                    <th>Inicio</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                {campaigns.length === 0 ? (
                                    <tr>
                                        <td colSpan={6} style={{ textAlign: 'center', padding: '3rem', color: 'var(--text-muted)' }}>
                                            <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>📣</div>
                                            <div>No hay campañas registradas</div>
                                            <button className="btn btn-primary" style={{ marginTop: '1rem' }} onClick={() => setIsModalOpen(true)}>
                                                ✨ Crear primera campaña
                                            </button>
                                        </td>
                                    </tr>
                                ) : (
                                    campaigns.map((campaign) => (
                                        <tr key={campaign.id}>
                                            <td>
                                                <div className="cell-main">{campaign.name}</div>
                                                <div className="cell-secondary">
                                                    {Array.isArray(campaign.platform)
                                                        ? campaign.platform.join(', ')
                                                        : campaign.platform}
                                                </div>
                                            </td>
                                            <td style={{ fontSize: '0.85rem' }}>
                                                {campaign.client?.name || 'Sin asignar'}
                                            </td>
                                            <td style={{ fontWeight: 600 }}>
                                                ${Number(campaign.budget).toLocaleString('es-ES', { minimumFractionDigits: 2 })}
                                            </td>
                                            <td>
                                                <span className={`badge ${getStatusConfig(campaign.status).class}`}>
                                                    <span className="badge-dot"></span>
                                                    {getStatusConfig(campaign.status).label}
                                                </span>
                                            </td>
                                            <td>
                                                {new Date(campaign.start_date).toLocaleDateString('es-ES', {
                                                    day: '2-digit',
                                                    month: 'short',
                                                    year: 'numeric'
                                                })}
                                            </td>
                                            <td>
                                                <div style={{ display: 'flex', gap: '0.5rem' }}>
                                                    <button className="btn-icon" title="Ver detalles">
                                                        👁️
                                                    </button>
                                                    <button className="btn-icon" title="Editar">
                                                        ✏️
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    ))
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            {/* Modal for creating new campaign */}
            {isModalOpen && (
                <div style={{
                    position: 'fixed',
                    inset: 0,
                    background: 'rgba(0, 0, 0, 0.7)',
                    backdropFilter: 'blur(4px)',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    zIndex: 9999,
                    padding: '1rem'
                }} onClick={() => setIsModalOpen(false)}>
                    <div className="card" style={{
                        maxWidth: '600px',
                        width: '100%',
                        maxHeight: '90vh',
                        overflowY: 'auto'
                    }} onClick={(e) => e.stopPropagation()}>
                        <div style={{ padding: '1.5rem', borderBottom: '1px solid var(--border-color)' }}>
                            <h2 style={{ fontSize: '1.5rem', fontWeight: 700, color: 'var(--text-primary)' }}>
                                ✨ Nueva Campaña
                            </h2>
                            <p style={{ fontSize: '0.9rem', color: 'var(--text-muted)', marginTop: '0.5rem' }}>
                                Crea una nueva campaña de marketing
                            </p>
                        </div>
                        <form onSubmit={handleCreate} style={{ padding: '1.5rem' }}>
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '1.25rem' }}>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Nombre de la campaña
                                    </label>
                                    <input
                                        type="text"
                                        className="input"
                                        value={formData.name}
                                        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                        required
                                        placeholder="Ej: Campaña Black Friday 2026"
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Cliente
                                    </label>
                                    <select
                                        className="input"
                                        value={formData.client_id}
                                        onChange={(e) => setFormData({ ...formData, client_id: e.target.value })}
                                    >
                                        <option value="">Seleccionar cliente</option>
                                        {clients.map(client => (
                                            <option key={client.id} value={client.id}>{client.name}</option>
                                        ))}
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Presupuesto
                                    </label>
                                    <input
                                        type="number"
                                        className="input"
                                        value={formData.budget}
                                        onChange={(e) => setFormData({ ...formData, budget: e.target.value })}
                                        required
                                        placeholder="0.00"
                                        step="0.01"
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Fecha de inicio
                                    </label>
                                    <input
                                        type="date"
                                        className="input"
                                        value={formData.start_date}
                                        onChange={(e) => setFormData({ ...formData, start_date: e.target.value })}
                                        required
                                    />
                                </div>
                                <div style={{ display: 'flex', gap: '0.75rem', justifyContent: 'flex-end', marginTop: '1rem' }}>
                                    <button type="button" className="btn btn-secondary" onClick={() => setIsModalOpen(false)}>
                                        Cancelar
                                    </button>
                                    <button type="submit" className="btn btn-primary">
                                        ✨ Crear Campaña
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
