import { useState } from 'react';

// ---------- Mock data (reemplazar por API) ----------
interface Service {
    id: number;
    name: string;
    provider: string;
    category: string; // dominio | hosting | …
    expiry_date: string;
    status: 'active' | 'expiring_soon' | 'expired';
}

const mockServices: Service[] = [
    {
        id: 1,
        name: 'acme-corp.com',
        provider: 'Namecheap',
        category: 'domains',
        expiry_date: '2026-01-28',
        status: 'expiring_soon',
    },
    {
        id: 2,
        name: 'Hosting Empresarial Pro',
        provider: 'Webuzo VPS',
        category: 'hosting',
        expiry_date: '2026-02-15',
        status: 'active',
    },
    // …añade más según necesites
];
// -----------------------------------------------------

export default function ServiceManagement() {
    const [services] = useState<Service[]>(mockServices);
    const [search, setSearch] = useState('');

    const filtered = services.filter(s =>
        s.name.toLowerCase().includes(search.toLowerCase()) ||
        s.provider.toLowerCase().includes(search.toLowerCase())
    );

    return (
        <div className="animate-fadeIn">
            {/* Header */}
            <div className="page-header">
                <div>
                    <h1 className="page-title">Gestión de Servicios</h1>
                    <p className="page-subtitle">Administra dominios, hosting y demás servicios</p>
                </div>
                <button className="btn btn-primary" onClick={() => alert('Abrir modal…')}>
                    ➕ Nuevo Servicio
                </button>
            </div>

            {/* Buscador */}
            <div className="filters" style={{ marginBottom: '1rem' }}>
                <input
                    type="text"
                    className="form-input"
                    placeholder="🔍 Buscar servicio o proveedor…"
                    value={search}
                    onChange={e => setSearch(e.target.value)}
                    style={{ minWidth: '250px' }}
                />
            </div>

            {/* Tabla */}
            <div className="table-container">
                <div className="table-wrapper">
                    <table className="table" style={{ tableLayout: 'fixed' }}>
                        <thead>
                            <tr>
                                <th style={{ width: '30%' }}>Servicio</th>
                                <th style={{ width: '20%' }}>Proveedor</th>
                                <th style={{ width: '18%' }}>Vencimiento</th>
                                <th style={{ width: '12%' }}>Estado</th>
                                <th style={{ width: '20%', textAlign: 'center' }}>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            {filtered.length === 0 ? (
                                <tr>
                                    <td colSpan={5}>
                                        <div className="empty-state">
                                            <div className="empty-state-icon">📋</div>
                                            <div className="empty-state-title">No hay servicios</div>
                                            <div className="empty-state-text">No se encontraron resultados con el filtro actual</div>
                                        </div>
                                    </td>
                                </tr>
                            ) : (
                                filtered.map(s => (
                                    <tr key={s.id}>
                                        <td>
                                            <div className="cell-main">{s.name}</div>
                                            <div className="cell-secondary">{s.category}</div>
                                        </td>
                                        <td>
                                            <div className="cell-main">{s.provider}</div>
                                        </td>
                                        <td>
                                            <div className="cell-main">
                                                {new Date(s.expiry_date).toLocaleDateString('es-ES', {
                                                    day: '2-digit',
                                                    month: 'short',
                                                    year: 'numeric',
                                                })}
                                            </div>
                                        </td>
                                        <td>
                                            <span
                                                className={`badge ${s.status === 'active'
                                                    ? 'badge-success'
                                                    : s.status === 'expiring_soon'
                                                        ? 'badge-warning'
                                                        : 'badge-danger'
                                                    }`}
                                            >
                                                <span className="badge-dot" />
                                                {s.status === 'active'
                                                    ? 'Activo'
                                                    : s.status === 'expiring_soon'
                                                        ? 'Por Vencer'
                                                        : 'Vencido'}
                                            </span>
                                        </td>
                                        <td>
                                            <div style={{ display: 'flex', gap: '0.35rem', justifyContent: 'center' }}>
                                                <button className="btn-icon" title="Editar" style={{ width: '28px', height: '28px' }}>
                                                    ✏️
                                                </button>
                                                <button className="btn-icon" title="Renovar" style={{ width: '28px', height: '28px' }}>
                                                    🔄
                                                </button>
                                                <button className="btn-icon" title="Eliminar" style={{ width: '28px', height: '28px' }}>
                                                    🗑️
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                ))
                            )}
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    );
}
