import { useEffect, useState, CSSProperties } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';
import NewProjectModal from '../../components/SoftwareDev/NewProjectModal';

interface DashboardMetrics {
    projects: { total: number; active: number; delivered: number };
    releases: { planned: number; in_qa: number; published_this_month: number };
    licenses: { active: number; expiring_soon: number; total_activations: number };
    recent_activity: any[];
}

export default function SoftwareDevDashboard() {
    const [stats, setStats] = useState<DashboardMetrics | null>(null);
    const [loading, setLoading] = useState(true);
    const [alertCount, setAlertCount] = useState(0);
    const [showNewProjectModal, setShowNewProjectModal] = useState(false);

    useEffect(() => {
        const fetchStats = async () => {
            try {
                const res = await axios.get('/api/software-dev/dashboard');
                setStats(res.data);
                // Calcular alertas
                const alerts = (res.data?.licenses?.expiring_soon || 0) + (res.data?.releases?.in_qa || 0);
                setAlertCount(alerts);
            } catch (error) {
                console.error('Error cargando métricas:', error);
            } finally {
                setLoading(false);
            }
        };
        fetchStats();
    }, []);

    const metricCards = [
        {
            icon: '💻',
            label: 'Proyectos Activos',
            value: (stats?.projects.active || 0).toString(),
            subtext: `${stats?.projects.total || 0} totales`,
            gradient: 'linear-gradient(135deg, #3b82f6, #2563eb)',
            glow: 'rgba(59, 130, 246, 0.4)',
        },
        {
            icon: '🐛',
            label: 'Releases en QA',
            value: (stats?.releases.in_qa || 0).toString(),
            subtext: `${stats?.releases.published_this_month || 0} publicados este mes`,
            gradient: 'linear-gradient(135deg, #f59e0b, #d97706)',
            glow: 'rgba(245, 158, 11, 0.4)',
        },
        {
            icon: '🔑',
            label: 'Licencias Activas',
            value: (stats?.licenses.active || 0).toString(),
            subtext: `${stats?.licenses.expiring_soon || 0} expiran pronto`,
            gradient: 'linear-gradient(135deg, #10b981, #059669)',
            glow: 'rgba(16, 185, 129, 0.4)',
            alert: (stats?.licenses.expiring_soon || 0) > 0,
        },
        {
            icon: '🌍',
            label: 'Total Activaciones',
            value: (stats?.licenses.total_activations || 0).toString(),
            subtext: 'Dispositivos únicos',
            gradient: 'linear-gradient(135deg, #8b5cf6, #7c3aed)',
            glow: 'rgba(139, 92, 246, 0.4)',
        },
    ];

    if (loading) {
        return (
            <div className="animate-fadeIn" style={{ display: 'flex', alignItems: 'center', justifyContent: 'center', minHeight: '400px' }}>
                <div style={{ textAlign: 'center', color: 'var(--text-muted)' }}>
                    <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>⏳</div>
                    <div>Cargando métricas...</div>
                </div>
            </div>
        );
    }

    return (
        <div className="animate-fadeIn">
            {/* Page Header */}
            <div className="page-header">
                <div>
                    <h1 className="page-title" style={{ display: 'flex', alignItems: 'center', gap: '0.75rem' }}>
                        💻 Desarrollo de Software (Desktop)
                        <span style={{
                            fontSize: '0.75rem',
                            padding: '0.25rem 0.75rem',
                            background: 'linear-gradient(135deg, #3b82f6, #2563eb)',
                            borderRadius: '8px',
                            fontWeight: 600,
                        }}>
                            SOFTWARE DEV
                        </span>
                    </h1>
                    <p className="page-subtitle">Gestión de ciclo de vida para aplicaciones de escritorio</p>
                </div>
                <div style={{ display: 'flex', gap: '0.75rem' }}>
                    <Link to="/app/software-dev/projects" className="btn btn-secondary">
                        📋 Ver Proyectos
                    </Link>
                    <button className="btn btn-primary" onClick={() => setShowNewProjectModal(true)}>
                        ➕ Nuevo Proyecto
                    </button>
                </div>
            </div>

            {/* Quick Stats Tags */}
            <div className="stats-row" style={{ display: 'flex', gap: '0.75rem', marginBottom: '2rem', flexWrap: 'wrap' }}>
                <div className="stat-tag">
                    <span>🗓️</span>
                    <span>Hoy:</span>
                    <span className="stat-tag-value">{new Date().toLocaleDateString('es-ES', { weekday: 'long', day: 'numeric', month: 'short' })}</span>
                </div>
                <div className="stat-tag">
                    <span>⚡</span>
                    <span>Alertas activas:</span>
                    <span className="stat-tag-value" style={{ color: alertCount > 0 ? 'var(--color-warning)' : 'inherit' }}>{alertCount}</span>
                </div>
                <div className="stat-tag">
                    <span>✅</span>
                    <span>Proyectos totales:</span>
                    <span className="stat-tag-value">{stats?.projects.total || 0}</span>
                </div>
            </div>

            {/* Metric Cards Grid */}
            <div className="cards-grid">
                {metricCards.map((card, index) => (
                    <div
                        key={index}
                        className="card metric-card"
                        style={{ '--card-glow': card.glow } as CSSProperties}
                    >
                        <div
                            className="metric-icon"
                            style={{ background: card.gradient }}
                        >
                            {card.icon}
                        </div>
                        <div className="metric-content">
                            <div className="metric-label">{card.label}</div>
                            <div className="metric-value">{card.value}</div>
                            {card.subtext && (
                                <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)', marginTop: '0.25rem' }}>
                                    {card.subtext}
                                </div>
                            )}
                            {card.alert && (
                                <div className="metric-change negative" style={{ background: 'rgba(239, 68, 68, 0.1)', color: 'var(--color-danger)' }}>
                                    ⚠️ Requiere atención
                                </div>
                            )}
                        </div>
                    </div>
                ))}
            </div>

            {/* Two Column Layout */}
            <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(400px, 1fr))', gap: '1.5rem', marginTop: '2.5rem' }}>
                {/* Recent Activity */}
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">📝 Actividad Reciente</h2>
                    </div>
                    <div style={{ padding: '0.5rem' }}>
                        {stats?.recent_activity && stats.recent_activity.length > 0 ? (
                            stats.recent_activity.slice(0, 5).map((log: any) => (
                                <div
                                    key={log.id}
                                    style={{
                                        display: 'flex',
                                        alignItems: 'flex-start',
                                        gap: '1rem',
                                        padding: '1rem',
                                        borderBottom: '1px solid var(--border-color)',
                                    }}
                                >
                                    <div style={{
                                        width: '40px',
                                        height: '40px',
                                        borderRadius: '10px',
                                        background: 'var(--glass-bg)',
                                        display: 'flex',
                                        alignItems: 'center',
                                        justifyContent: 'center',
                                        fontSize: '0.9rem',
                                        fontWeight: 700,
                                        color: 'var(--text-muted)',
                                    }}>
                                        {log.actor?.name?.substring(0, 2).toUpperCase() || 'SYS'}
                                    </div>
                                    <div style={{ flex: 1 }}>
                                        <div style={{ fontWeight: 500, color: 'var(--text-primary)', marginBottom: '0.25rem' }}>
                                            <span>{log.actor?.name || 'Sistema'}</span> realizó <strong>{log.action}</strong> en{' '}
                                            <span style={{ color: 'var(--color-primary)' }}>{log.entity}</span>
                                        </div>
                                        <div style={{ fontSize: '0.85rem', color: 'var(--text-muted)' }}>
                                            {new Date(log.created_at).toLocaleDateString('es-ES', {
                                                day: '2-digit',
                                                month: 'short',
                                                hour: '2-digit',
                                                minute: '2-digit'
                                            })}
                                        </div>
                                    </div>
                                </div>
                            ))
                        ) : (
                            <div style={{ padding: '2rem', textAlign: 'center', color: 'var(--text-muted)' }}>
                                <div>🔔</div>
                                <div>No hay actividad reciente registrada.</div>
                            </div>
                        )}
                    </div>
                </div>

                {/* Quick Actions */}
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">⚡ Accesos Rápidos</h2>
                    </div>
                    <div style={{ padding: '0.5rem' }}>
                        <Link
                            to="/app/software-dev/plans"
                            style={{
                                display: 'flex',
                                alignItems: 'center',
                                gap: '1rem',
                                padding: '1rem',
                                borderBottom: '1px solid var(--border-color)',
                                textDecoration: 'none',
                                color: 'inherit',
                                transition: 'background 0.2s ease',
                            }}
                        >
                            <div style={{
                                width: '48px',
                                height: '48px',
                                borderRadius: '12px',
                                background: 'linear-gradient(135deg, #ec4899, #db2777)',
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'center',
                                fontSize: '1.5rem',
                            }}>
                                📋
                            </div>
                            <div>
                                <div style={{ fontWeight: 600, color: 'var(--text-primary)' }}>Gestión de Planes</div>
                                <div style={{ fontSize: '0.85rem', color: 'var(--text-muted)' }}>Configurar precios y features</div>
                            </div>
                        </Link>

                        <Link
                            to="#"
                            onClick={(e) => { e.preventDefault(); alert('Próximamente'); }}
                            style={{
                                display: 'flex',
                                alignItems: 'center',
                                gap: '1rem',
                                padding: '1rem',
                                borderBottom: '1px solid var(--border-color)',
                                textDecoration: 'none',
                                color: 'inherit',
                            }}
                        >
                            <div style={{
                                width: '48px',
                                height: '48px',
                                borderRadius: '12px',
                                background: 'linear-gradient(135deg, #f59e0b, #d97706)',
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'center',
                                fontSize: '1.5rem',
                            }}>
                                🚀
                            </div>
                            <div>
                                <div style={{ fontWeight: 600, color: 'var(--text-primary)' }}>Publicar Release</div>
                                <div style={{ fontSize: '0.85rem', color: 'var(--text-muted)' }}>Subir nueva versión</div>
                            </div>
                        </Link>

                        <Link
                            to="#"
                            onClick={(e) => { e.preventDefault(); alert('Próximamente'); }}
                            style={{
                                display: 'flex',
                                alignItems: 'center',
                                gap: '1rem',
                                padding: '1rem',
                                textDecoration: 'none',
                                color: 'inherit',
                            }}
                        >
                            <div style={{
                                width: '48px',
                                height: '48px',
                                borderRadius: '12px',
                                background: 'linear-gradient(135deg, #10b981, #059669)',
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'center',
                                fontSize: '1.5rem',
                            }}>
                                🎟️
                            </div>
                            <div>
                                <div style={{ fontWeight: 600, color: 'var(--text-primary)' }}>Generar Licencias</div>
                                <div style={{ fontSize: '0.85rem', color: 'var(--text-muted)' }}>Crear claves por lote</div>
                            </div>
                        </Link>
                    </div>
                </div>
            </div>

            {/* Modal Nuevo Proyecto */}
            <NewProjectModal
                isOpen={showNewProjectModal}
                onClose={() => setShowNewProjectModal(false)}
                onSuccess={() => {
                    // Recargar datos del dashboard
                    const fetchStats = async () => {
                        try {
                            const res = await axios.get('/api/software-dev/dashboard');
                            setStats(res.data);
                            const alerts = (res.data?.licenses?.expiring_soon || 0) + (res.data?.releases?.in_qa || 0);
                            setAlertCount(alerts);
                        } catch (error) {
                            console.error('Error recargando dashboard:', error);
                        }
                    };
                    fetchStats();
                }}
            />
        </div>
    );
}
