import React, { useEffect, useState } from 'react';
import axios from 'axios';
import PageHeader from '../../components/PageHeader';
import { Link } from 'react-router-dom';

export default function SoftwareDevProjects() {
    const [projects, setProjects] = useState([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        loadProjects();
    }, []);

    const loadProjects = async () => {
        try {
            const res = await axios.get('/api/software-dev/projects');
            setProjects(res.data); // Asumiendo que devuelve array directo según AppDevProjectController
        } catch (error) {
            console.error('Error cargando proyectos:', error);
        } finally {
            setLoading(false);
        }
    };

    const getStatusBadge = (status: string) => {
        const colors: any = {
            discovery: 'bg-purple-100 text-purple-800',
            in_progress: 'bg-blue-100 text-blue-800',
            qa: 'bg-orange-100 text-orange-800',
            delivered: 'bg-green-100 text-green-800',
            maintenance: 'bg-teal-100 text-teal-800',
            closed: 'bg-gray-100 text-gray-800',
        };
        return <span className={`badge ${colors[status] || 'bg-gray-100'}`}>{status.replace('_', ' ')}</span>;
    };

    return (
        <div className="animate-fade-in space-y-6">
            <PageHeader
                title="Proyectos de Software"
                subtitle="Listado de aplicaciones de escritorio en desarrollo y mantenimiento"
                actions={
                    <button className="btn btn-primary" onClick={() => alert('Próximamente: Modal Crear Proyecto')}>
                        + Nuevo Proyecto
                    </button>
                }
            />

            <div className="card overflow-hidden">
                <div className="table-container">
                    <table className="table">
                        <thead>
                            <tr>
                                <th>Código</th>
                                <th>Proyecto</th>
                                <th>Cliente</th>
                                <th>Plataforma</th>
                                <th>Estado</th>
                                <th>Inicio</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            {loading ? (
                                <tr>
                                    <td colSpan={7} className="text-center py-8 text-gray-500">Cargando...</td>
                                </tr>
                            ) : projects.length === 0 ? (
                                <tr>
                                    <td colSpan={7} className="text-center py-8 text-gray-500">
                                        No hay proyectos registrados.
                                    </td>
                                </tr>
                            ) : (
                                projects.map((project: any) => (
                                    <tr key={project.id}>
                                        <td className="font-mono text-xs">{project.code}</td>
                                        <td>
                                            <div className="font-medium text-gray-900 dark:text-white">
                                                {project.name}
                                            </div>
                                            <div className="text-xs text-gray-500">
                                                {project.plan?.name || 'Sin plan'}
                                            </div>
                                        </td>
                                        <td>{project.client?.name || 'N/A'}</td>
                                        <td>
                                            <span className="capitalize">{project.platform}</span>
                                        </td>
                                        <td>{getStatusBadge(project.status)}</td>
                                        <td>{project.start_date}</td>
                                        <td>
                                            <Link
                                                to={`/app/software-dev/projects/${project.id}`}
                                                className="text-blue-600 hover:text-blue-800 font-medium"
                                            >
                                                Gestionar
                                            </Link>
                                        </td>
                                    </tr>
                                ))
                            )}
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    );
}
