import { useState, useEffect } from 'react';
import { useSearchParams } from 'react-router-dom';
import axios from 'axios';
import PageHeader from '../../components/PageHeader';

interface Product {
    id: number;
    sku: string;
    name: string;
    category: string;
    brand?: string;
    model?: string;
    price: number;
    cost: number;
    stock: number;
    reorder_level: number;
    is_active: boolean;
}

const categoryConfig: Record<string, { label: string; emoji: string; color: string }> = {
    laptop: { label: 'Laptop', emoji: '💻', color: '#667eea' },
    desktop: { label: 'Desktop', emoji: '🖥️', color: '#764ba2' },
    monitor: { label: 'Monitor', emoji: '🖥️', color: '#30cfd0' },
    printer: { label: 'Impresora', emoji: '🖨️', color: '#f093fb' },
    networking: { label: 'Redes', emoji: '📡', color: '#4facfe' },
    parts: { label: 'Partes', emoji: '🔧', color: '#43e97b' },
    accessories: { label: 'Accesorios', emoji: '⌨️', color: '#fa709a' },
    other: { label: 'Otro', emoji: '📦', color: '#a18cd1' },
};

export default function StoreProducts() {
    const [searchParams] = useSearchParams();
    const [products, setProducts] = useState<Product[]>([]);
    const [loading, setLoading] = useState(true);
    const [filter, setFilter] = useState({
        category: searchParams.get('category') || '',
        low_stock: searchParams.get('filter') === 'low_stock',
    });

    useEffect(() => {
        fetchProducts();
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [filter.category, filter.low_stock]);

    const fetchProducts = async () => {
        try {
            setLoading(true);
            const params = new URLSearchParams();
            if (filter.category) params.append('category', filter.category);
            if (filter.low_stock) params.append('low_stock', '1');

            const res = await axios.get(`/api/store/products?${params}`);
            setProducts(res.data.data || res.data);
        } catch (error) {
            console.error('Error fetching products:', error);
        } finally {
            setLoading(false);
        }
    };

    if (loading) {
        return <div className="loading">Cargando...</div>;
    }

    return (
        <div className="page">
            <PageHeader
                title="📦 Catálogo de Productos"
                subtitle="Gestión de inventario y precios"
                actions={
                    <button className="btn btn-primary" onClick={() => alert('Agregar producto próximamente')}>
                        ➕ Nuevo Producto
                    </button>
                }
            />

            {/* Filtros */}
            <div className="card" style={{ marginBottom: '1.5rem' }}>
                <div style={{ display: 'flex', gap: '1rem', flexWrap: 'wrap', alignItems: 'flex-end' }}>
                    <div>
                        <label className="form-label" style={{ marginBottom: '0.5rem' }}>Categoría</label>
                        <select
                            className="input"
                            value={filter.category}
                            onChange={(e) => setFilter({ ...filter, category: e.target.value })}
                            style={{ minWidth: '180px' }}
                        >
                            <option value="">Todas</option>
                            <option value="laptop">💻 Laptops</option>
                            <option value="desktop">🖥️ Desktops</option>
                            <option value="monitor">🖥️ Monitores</option>
                            <option value="printer">🖨️ Impresoras</option>
                            <option value="networking">📡 Redes</option>
                            <option value="parts">🔧 Partes</option>
                            <option value="accessories">⌨️ Accesorios</option>
                        </select>
                    </div>
                    <div>
                        <label className="form-label" style={{ marginBottom: '0.5rem' }}>
                            <input
                                type="checkbox"
                                checked={filter.low_stock}
                                onChange={(e) => setFilter({ ...filter, low_stock: e.target.checked })}
                                style={{ marginRight: '0.5rem' }}
                            />
                            Solo Stock Bajo
                        </label>
                    </div>
                    {(filter.category || filter.low_stock) && (
                        <button
                            className="btn btn-ghost"
                            onClick={() => setFilter({ category: '', low_stock: false })}
                        >
                            🔄 Limpiar Filtros
                        </button>
                    )}
                </div>
            </div>

            {/* Grid de Productos */}
            <div style={{
                display: 'grid',
                gridTemplateColumns: 'repeat(auto-fill, minmax(300px, 1fr))',
                gap: '1.5rem'
            }}>
                {products.length === 0 ? (
                    <div className="card" style={{ gridColumn: '1 / -1', padding: '3rem', textAlign: 'center' }}>
                        <div style={{ fontSize: '3rem', marginBottom: '1rem' }}>📦</div>
                        <div style={{ color: 'var(--text-secondary)', fontSize: '1.1rem' }}>
                            No hay productos registrados
                        </div>
                    </div>
                ) : (
                    products.map((product) => {
                        const catInfo = categoryConfig[product.category] || categoryConfig.other;
                        const stockPercent = (product.stock / (product.reorder_level * 2)) * 100;
                        const isLowStock = product.stock <= product.reorder_level;

                        // Convertir a números por si vienen como strings del API
                        const price = parseFloat(String(product.price));
                        const cost = parseFloat(String(product.cost));
                        const margin = ((price - cost) / cost) * 100;

                        return (
                            <div
                                key={product.id}
                                className="card"
                                style={{
                                    padding: '1.5rem',
                                    borderLeft: `4px solid ${catInfo.color}`,
                                    opacity: product.is_active ? 1 : 0.6,
                                }}
                            >
                                {/* Header */}
                                <div style={{ display: 'flex', alignItems: 'flex-start', gap: '1rem', marginBottom: '1rem' }}>
                                    <div
                                        style={{
                                            width: '48px',
                                            height: '48px',
                                            borderRadius: '12px',
                                            background: `linear-gradient(135deg, ${catInfo.color}20, ${catInfo.color}10)`,
                                            display: 'flex',
                                            alignItems: 'center',
                                            justifyContent: 'center',
                                            fontSize: '1.5rem',
                                        }}
                                    >
                                        {catInfo.emoji}
                                    </div>
                                    <div style={{ flex: 1 }}>
                                        <div style={{ fontSize: '0.875rem', color: 'var(--text-secondary)' }}>
                                            {product.sku}
                                        </div>
                                        <div style={{ fontSize: '1.05rem', fontWeight: 700, marginTop: '0.25rem' }}>
                                            {product.name}
                                        </div>
                                    </div>
                                </div>

                                {/* Marca/Modelo */}
                                {product.brand && (
                                    <div style={{ fontSize: '0.875rem', color: 'var(--text-secondary)', marginBottom: '0.75rem' }}>
                                        {product.brand} {product.model}
                                    </div>
                                )}

                                {/* Stock */}
                                <div style={{ marginBottom: '1rem' }}>
                                    <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '0.5rem' }}>
                                        <span style={{ fontSize: '0.875rem', color: 'var(--text-secondary)' }}>Stock</span>
                                        <span style={{
                                            fontWeight: 700,
                                            color: isLowStock ? '#ef4444' : '#10b981'
                                        }}>
                                            {product.stock} {isLowStock && '⚠️'}
                                        </span>
                                    </div>
                                    <div style={{
                                        height: '6px',
                                        background: 'var(--glass-bg)',
                                        borderRadius: '999px',
                                        overflow: 'hidden',
                                    }}>
                                        <div style={{
                                            height: '100%',
                                            width: `${Math.min(stockPercent, 100)}%`,
                                            background: isLowStock
                                                ? 'linear-gradient(90deg, #ef4444, #f59e0b)'
                                                : 'linear-gradient(90deg, #10b981, #43e97b)',
                                            transition: 'width 0.3s ease',
                                        }} />
                                    </div>
                                </div>

                                {/* Precio */}
                                <div style={{
                                    display: 'flex',
                                    justifyContent: 'space-between',
                                    padding: '0.75rem',
                                    background: 'var(--glass-bg)',
                                    borderRadius: '8px',
                                    marginBottom: '1rem',
                                }}>
                                    <div>
                                        <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)' }}>Precio</div>
                                        <div style={{ fontSize: '1.25rem', fontWeight: 700, color: 'var(--color-primary)' }}>
                                            ${price.toFixed(2)}
                                        </div>
                                    </div>
                                    <div style={{ textAlign: 'right' }}>
                                        <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)' }}>Margen</div>
                                        <div style={{ fontSize: '1rem', fontWeight: 600, color: '#10b981' }}>
                                            {margin.toFixed(0)}%
                                        </div>
                                    </div>
                                </div>

                                {/* Acciones */}
                                <div style={{ display: 'flex', gap: '0.5rem' }}>
                                    <button
                                        className="btn btn-sm btn-secondary"
                                        onClick={() => alert('Editar próximamente')}
                                        style={{ flex: 1 }}
                                    >
                                        ✏️ Editar
                                    </button>
                                    <button
                                        className="btn btn-sm btn-primary"
                                        onClick={() => alert('Ajustar stock próximamente')}
                                        style={{ flex: 1 }}
                                    >
                                        📊 Stock
                                    </button>
                                </div>
                            </div>
                        );
                    })
                )}
            </div>

            {/* Resumen */}
            {products.length > 0 && (
                <div className="card" style={{ marginTop: '2rem', textAlign: 'center' }}>
                    <div style={{ color: 'var(--text-secondary)' }}>
                        📦 Mostrando <strong>{products.length}</strong> productos
                    </div>
                </div>
            )}
        </div>
    );
}
