import { useState, useMemo, CSSProperties } from 'react';
import { Link } from 'react-router-dom';
import PageHeader from '../components/PageHeader';

interface Ticket {
    id: number;
    subject: string;
    client_name: string;
    priority: 'low' | 'medium' | 'high' | 'critical';
    status: 'open' | 'in_progress' | 'waiting' | 'resolved';
    created_at: string;
    updated_at: string;
    assigned_to?: string;
}

/**
 * 🎫 PÁGINA: SOPORTE TÉCNICO (TICKETS)
 * 
 * ✅ ESTILO ACTUALIZADO - Usa el diseño premium estándar
 */

// Mock data
const mockTickets: Ticket[] = [
    { id: 1, subject: 'Página web no carga correctamente', client_name: 'Acme Corporation', priority: 'critical', status: 'open', created_at: '2026-01-25T10:30:00', updated_at: '2026-01-25T10:30:00', assigned_to: 'Juan Pérez' },
    { id: 2, subject: 'Error en formulario de contacto', client_name: 'Beta Technologies', priority: 'high', status: 'in_progress', created_at: '2026-01-24T14:20:00', updated_at: '2026-01-26T09:15:00', assigned_to: 'María López' },
    { id: 3, subject: 'Solicitud de backup de base de datos', client_name: 'Gamma Retail SA', priority: 'medium', status: 'waiting', created_at: '2026-01-23T09:15:00', updated_at: '2026-01-24T16:00:00' },
    { id: 4, subject: 'Actualizar certificado SSL', client_name: 'Delta Industries', priority: 'high', status: 'open', created_at: '2026-01-25T08:00:00', updated_at: '2026-01-25T08:00:00' },
    { id: 5, subject: 'Configurar correos corporativos', client_name: 'Epsilon Tech', priority: 'low', status: 'in_progress', created_at: '2026-01-22T16:45:00', updated_at: '2026-01-26T11:30:00', assigned_to: 'Carlos Ruiz' },
    { id: 6, subject: 'Migración de hosting', client_name: 'Zeta Media', priority: 'medium', status: 'resolved', created_at: '2026-01-20T10:00:00', updated_at: '2026-01-24T18:00:00', assigned_to: 'Ana García' },
    { id: 7, subject: 'Optimización de velocidad del sitio', client_name: 'Sigma Corp', priority: 'low', status: 'resolved', created_at: '2026-01-18T14:30:00', updated_at: '2026-01-22T12:00:00', assigned_to: 'Juan Pérez' },
];

const priorityConfig: Record<string, { label: string; color: string; emoji: string }> = {
    low: { label: 'Baja', color: '#64748b', emoji: '🟢' },
    medium: { label: 'Media', color: '#f59e0b', emoji: '🟡' },
    high: { label: 'Alta', color: '#ef4444', emoji: '🟠' },
    critical: { label: 'Crítica', color: '#dc2626', emoji: '🔴' },
};

const statusConfig: Record<string, { label: string; class: string }> = {
    open: { label: 'Abierto', class: 'badge-danger' },
    in_progress: { label: 'En Progreso', class: 'badge-primary' },
    waiting: { label: 'Esperando', class: 'badge-warning' },
    resolved: { label: 'Resuelto', class: 'badge-success' },
};

export default function Tickets() {
    const [tickets, setTickets] = useState<Ticket[]>(mockTickets);
    const [search, setSearch] = useState('');
    const [filterStatus, setFilterStatus] = useState<string>('all');
    const [filterPriority, setFilterPriority] = useState<string>('all');
    const [showNewTicketModal, setShowNewTicketModal] = useState(false);

    const filteredTickets = useMemo(() => {
        return tickets.filter(ticket => {
            const matchSearch = ticket.subject.toLowerCase().includes(search.toLowerCase()) ||
                ticket.client_name.toLowerCase().includes(search.toLowerCase());
            const matchStatus = filterStatus === 'all' || ticket.status === filterStatus;
            const matchPriority = filterPriority === 'all' || ticket.priority === filterPriority;
            return matchSearch && matchStatus && matchPriority;
        });
    }, [tickets, search, filterStatus, filterPriority]);

    // Stats
    const stats = useMemo(() => ({
        total: tickets.length,
        open: tickets.filter(t => t.status === 'open').length,
        in_progress: tickets.filter(t => t.status === 'in_progress').length,
        waiting: tickets.filter(t => t.status === 'waiting').length,
        resolved: tickets.filter(t => t.status === 'resolved').length,
        critical: tickets.filter(t => t.priority === 'critical').length,
    }), [tickets]);

    const metricCards = [
        {
            icon: '🎫',
            label: 'Total Tickets',
            value: stats.total.toString(),
            subtext: `${stats.open + stats.in_progress} activos`,
            gradient: 'linear-gradient(135deg, #ec4899, #db2777)',
            glow: 'rgba(236, 72, 153, 0.4)',
        },
        {
            icon: '🚨',
            label: 'Abiertos',
            value: stats.open.toString(),
            gradient: 'linear-gradient(135deg, #ef4444, #dc2626)',
            glow: 'rgba(239, 68, 68, 0.4)',
            alert: stats.open > 5,
        },
        {
            icon: '⚙️',
            label: 'En Progreso',
            value: stats.in_progress.toString(),
            gradient: 'linear-gradient(135deg, #3b82f6, #2563eb)',
            glow: 'rgba(59, 130, 246, 0.4)',
        },
        {
            icon: '✅',
            label: 'Resueltos',
            value: stats.resolved.toString(),
            gradient: 'linear-gradient(135deg, #10b981, #059669)',
            glow: 'rgba(16, 185, 129, 0.4)',
        },
    ];

    return (
        <div className="animate-fadeIn">
            {/* Page Header */}
            <PageHeader
                title="Soporte Técnico"
                subtitle="Gestiona los tickets de soporte de tus clientes"
                actions={
                    <>
                        <Link to="/app/tickets/reports" className="btn btn-secondary">
                            📊 Reportes
                        </Link>
                        <button className="btn btn-primary" onClick={() => setShowNewTicketModal(true)}>
                            ➕ Nuevo Ticket
                        </button>
                    </>
                }
                stats={[
                    { icon: '📅', label: 'Hoy', value: new Date().toLocaleDateString('es-ES', { weekday: 'long', day: 'numeric', month: 'short' }) },
                    { icon: '🔴', label: 'Críticos', value: stats.critical, color: 'var(--color-danger)' },
                ]}
            />

            {/* Metric Cards Grid */}
            <div className="cards-grid">
                {metricCards.map((card, index) => (
                    <div
                        key={index}
                        className="card metric-card"
                        style={{ '--card-glow': card.glow } as CSSProperties}
                    >
                        <div
                            className="metric-icon"
                            style={{ background: card.gradient }}
                        >
                            {card.icon}
                        </div>
                        <div className="metric-content">
                            <div className="metric-label">{card.label}</div>
                            <div className="metric-value">{card.value}</div>
                            {card.subtext && (
                                <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)', marginTop: '0.25rem' }}>
                                    {card.subtext}
                                </div>
                            )}
                            {card.alert && (
                                <div className="metric-change negative" style={{ background: 'rgba(239, 68, 68, 0.1)', color: 'var(--color-danger)' }}>
                                    ⚠️ Requiere atención
                                </div>
                            )}
                        </div>
                    </div>
                ))}
            </div>

            {/* Filters */}
            <div className="filters" style={{ marginTop: '2rem' }}>
                <div className="form-group" style={{ marginBottom: 0, flex: 1, maxWidth: '350px' }}>
                    <input
                        type="text"
                        className="form-input"
                        placeholder="🔍 Buscar por asunto o cliente..."
                        value={search}
                        onChange={(e) => setSearch(e.target.value)}
                    />
                </div>

                <select
                    className="form-select"
                    value={filterStatus}
                    onChange={(e) => setFilterStatus(e.target.value)}
                    style={{ minWidth: '150px' }}
                >
                    <option value="all">Todos los estados</option>
                    <option value="open">🚨 Abiertos</option>
                    <option value="in_progress">⚙️ En Progreso</option>
                    <option value="waiting">⏸️ Esperando</option>
                    <option value="resolved">✅ Resueltos</option>
                </select>

                <select
                    className="form-select"
                    value={filterPriority}
                    onChange={(e) => setFilterPriority(e.target.value)}
                    style={{ minWidth: '150px' }}
                >
                    <option value="all">Todas las prioridades</option>
                    <option value="critical">🔴 Crítica</option>
                    <option value="high">🟠 Alta</option>
                    <option value="medium">🟡 Media</option>
                    <option value="low">🟢 Baja</option>
                </select>

                {(search || filterStatus !== 'all' || filterPriority !== 'all') && (
                    <button
                        className="btn btn-ghost btn-sm"
                        onClick={() => { setSearch(''); setFilterStatus('all'); setFilterPriority('all'); }}
                    >
                        ✕ Limpiar filtros
                    </button>
                )}
            </div>

            {/* Tickets Table */}
            <div style={{ marginTop: '1.5rem' }}>
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">🎫 Tickets de Soporte</h2>
                        <button className="btn btn-ghost btn-sm">
                            🔄 Actualizar
                        </button>
                    </div>
                    <div className="table-wrapper">
                        <table className="table">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Asunto</th>
                                    <th>Cliente</th>
                                    <th>Prioridad</th>
                                    <th>Estado</th>
                                    <th>Asignado a</th>
                                    <th>Creado</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                {filteredTickets.length === 0 ? (
                                    <tr>
                                        <td colSpan={8} style={{ textAlign: 'center', padding: '3rem', color: 'var(--text-muted)' }}>
                                            <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>🎫</div>
                                            <div>No hay tickets que coincidan con los filtros</div>
                                        </td>
                                    </tr>
                                ) : (
                                    filteredTickets.map((ticket) => (
                                        <tr key={ticket.id}>
                                            <td>
                                                <span style={{ fontWeight: 700, color: 'var(--text-primary)' }}>
                                                    #{ticket.id}
                                                </span>
                                            </td>
                                            <td>
                                                <div className="cell-main" style={{ maxWidth: '300px' }}>
                                                    {ticket.subject}
                                                </div>
                                            </td>
                                            <td style={{ fontSize: '0.85rem' }}>{ticket.client_name}</td>
                                            <td>
                                                <span style={{
                                                    display: 'inline-flex',
                                                    alignItems: 'center',
                                                    gap: '0.25rem',
                                                    padding: '0.2rem 0.6rem',
                                                    borderRadius: '6px',
                                                    fontSize: '0.75rem',
                                                    fontWeight: 600,
                                                    background: `${priorityConfig[ticket.priority].color}15`,
                                                    color: priorityConfig[ticket.priority].color,
                                                }}>
                                                    {priorityConfig[ticket.priority].emoji}
                                                    {priorityConfig[ticket.priority].label}
                                                </span>
                                            </td>
                                            <td>
                                                <span className={`badge ${statusConfig[ticket.status].class}`}>
                                                    <span className="badge-dot"></span>
                                                    {statusConfig[ticket.status].label}
                                                </span>
                                            </td>
                                            <td style={{ fontSize: '0.85rem' }}>
                                                {ticket.assigned_to || <span style={{ color: 'var(--text-muted)' }}>Sin asignar</span>}
                                            </td>
                                            <td>
                                                {new Date(ticket.created_at).toLocaleDateString('es-ES', {
                                                    day: '2-digit',
                                                    month: 'short',
                                                    hour: '2-digit',
                                                    minute: '2-digit'
                                                })}
                                            </td>
                                            <td>
                                                <div style={{ display: 'flex', gap: '0.5rem' }}>
                                                    <button className="btn-icon" title="Ver detalles">
                                                        👁️
                                                    </button>
                                                    <button className="btn-icon" title="Editar">
                                                        ✏️
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    ))
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            {/* Results Count */}
            <div style={{
                marginTop: '1rem',
                fontSize: '0.85rem',
                color: 'var(--text-muted)',
                display: 'flex',
                justifyContent: 'space-between',
                alignItems: 'center'
            }}>
                <span>Mostrando {filteredTickets.length} de {tickets.length} tickets</span>
                <button className="btn btn-ghost btn-sm">📥 Exportar</button>
            </div>

            {/* New Ticket Modal */}
            {showNewTicketModal && (
                <div style={{
                    position: 'fixed',
                    inset: 0,
                    background: 'rgba(0, 0, 0, 0.7)',
                    backdropFilter: 'blur(4px)',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    zIndex: 9999,
                    padding: '1rem'
                }} onClick={() => setShowNewTicketModal(false)}>
                    <div className="card" style={{
                        maxWidth: '600px',
                        width: '100%',
                        maxHeight: '90vh',
                        overflowY: 'auto'
                    }} onClick={(e) => e.stopPropagation()}>
                        <div style={{ padding: '1.5rem', borderBottom: '1px solid var(--border-color)' }}>
                            <h2 style={{ fontSize: '1.5rem', fontWeight: 700, color: 'var(--text-primary)' }}>
                                🎫 Nuevo Ticket
                            </h2>
                            <p style={{ fontSize: '0.9rem', color: 'var(--text-muted)', marginTop: '0.5rem' }}>
                                Crea un nuevo ticket de soporte
                            </p>
                        </div>
                        <form style={{ padding: '1.5rem' }}>
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '1.25rem' }}>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Cliente
                                    </label>
                                    <select className="input" required>
                                        <option value="">Seleccionar cliente</option>
                                        <option>Acme Corporation</option>
                                        <option>Beta Technologies</option>
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Asunto
                                    </label>
                                    <input
                                        type="text"
                                        className="input"
                                        required
                                        placeholder="Describe brevemente el problema"
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Prioridad
                                    </label>
                                    <select className="input" required>
                                        <option value="low">🟢 Baja</option>
                                        <option value="medium">🟡 Media</option>
                                        <option value="high">🟠 Alta</option>
                                        <option value="critical">🔴 Crítica</option>
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Descripción
                                    </label>
                                    <textarea
                                        className="input"
                                        rows={4}
                                        required
                                        placeholder="Describe el problema en detalle..."
                                    />
                                </div>
                                <div style={{ display: 'flex', gap: '0.75rem', justifyContent: 'flex-end', marginTop: '1rem' }}>
                                    <button type="button" className="btn btn-secondary" onClick={() => setShowNewTicketModal(false)}>
                                        Cancelar
                                    </button>
                                    <button type="submit" className="btn btn-primary">
                                        🎫 Crear Ticket
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
