import { useState, useEffect, CSSProperties } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';
import PageHeader from '../../components/PageHeader';

/**
 * 🎨 PÁGINA: DISEÑO WEB - DASHBOARD
 * 
 * ✅ ESTILO ACTUALIZADO - Mismo diseño premium que Marketing
 */

export default function WebDesignDashboard() {
    const [projects, setProjects] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        fetchProjects();
    }, []);

    const fetchProjects = async () => {
        try {
            const res = await axios.get('/api/web-design/projects');
            setProjects(res.data);
        } catch (error) {
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    // Calculate metrics
    const activeProjects = projects.filter(p => p.status === 'in_progress').length;
    const completedProjects = projects.filter(p => p.status === 'completed').length;
    const totalBudget = projects.reduce((acc, p) => acc + Number(p.budget || 0), 0);

    const metricCards = [
        {
            icon: '🎨',
            label: 'Proyectos Activos',
            value: activeProjects.toString(),
            subtext: `${projects.length} totales`,
            gradient: 'linear-gradient(135deg, #ec4899, #db2777)',
            glow: 'rgba(236, 72, 153, 0.4)',
        },
        {
            icon: '💰',
            label: 'Presupuesto Total',
            value: `$${totalBudget.toLocaleString('es-ES', { minimumFractionDigits: 2 })}`,
            gradient: 'linear-gradient(135deg, #3b82f6, #2563eb)',
            glow: 'rgba(59, 130, 246, 0.4)',
        },
        {
            icon: '✅',
            label: 'Completados',
            value: completedProjects.toString(),
            gradient: 'linear-gradient(135deg, #10b981, #059669)',
            glow: 'rgba(16, 185, 129, 0.4)',
        },
    ];

    if (loading) {
        return (
            <div className="animate-fadeIn" style={{ display: 'flex', alignItems: 'center', justifyContent: 'center', minHeight: '400px' }}>
                <div style={{ textAlign: 'center', color: 'var(--text-muted)' }}>
                    <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>⏳</div>
                    <div>Cargando proyectos...</div>
                </div>
            </div>
        );
    }

    return (
        <div className="animate-fadeIn">
            <PageHeader
                title="Diseño Web"
                subtitle="Gestiona tus proyectos de diseño y desarrollo web"
                actions={
                    <Link to="/app/web-design/projects" className="btn btn-primary">
                        🎨 Ver Proyectos
                    </Link>
                }
                stats={[
                    { icon: '📅', label: 'Hoy', value: new Date().toLocaleDateString('es-ES', { weekday: 'long', day: 'numeric', month: 'short' }) },
                    { icon: '📊', label: 'Total proyectos', value: projects.length },
                ]}
            />

            {/* Metric Cards Grid */}
            <div className="cards-grid">
                {metricCards.map((card, index) => (
                    <div
                        key={index}
                        className="card metric-card"
                        style={{ '--card-glow': card.glow } as CSSProperties}
                    >
                        <div
                            className="metric-icon"
                            style={{ background: card.gradient }}
                        >
                            {card.icon}
                        </div>
                        <div className="metric-content">
                            <div className="metric-label">{card.label}</div>
                            <div className="metric-value">{card.value}</div>
                            {card.subtext && (
                                <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)', marginTop: '0.25rem' }}>
                                    {card.subtext}
                                </div>
                            )}
                        </div>
                    </div>
                ))}
            </div>

            {/* Projects Table */}
            <div style={{ marginTop: '2.5rem' }}>
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">🎨 Proyectos Recientes</h2>
                        <Link to="/app/web-design/projects" className="btn btn-ghost btn-sm">
                            Ver todos →
                        </Link>
                    </div>
                    <div className="table-wrapper">
                        <table className="table">
                            <thead>
                                <tr>
                                    <th>Proyecto</th>
                                    <th>Cliente</th>
                                    <th>Estado</th>
                                    <th>Presupuesto</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                {projects.length === 0 ? (
                                    <tr>
                                        <td colSpan={5} style={{ textAlign: 'center', padding: '3rem', color: 'var(--text-muted)' }}>
                                            <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>🎨</div>
                                            <div>No hay proyectos registrados</div>
                                        </td>
                                    </tr>
                                ) : (
                                    projects.slice(0, 5).map((project) => (
                                        <tr key={project.id}>
                                            <td>
                                                <div style={{ fontWeight: 600 }}>{project.name}</div>
                                                {project.url && <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)' }}>{project.url}</div>}
                                            </td>
                                            <td>{project.client?.name || 'N/A'}</td>
                                            <td>
                                                <span className={`badge ${project.status === 'completed' ? 'badge-success' :
                                                        project.status === 'in_progress' ? 'badge-primary' :
                                                            'badge-warning'
                                                    }`}>
                                                    {project.status}
                                                </span>
                                            </td>
                                            <td>${Number(project.budget || 0).toLocaleString('es-ES')}</td>
                                            <td>
                                                <Link to={`/app/web-design/projects/${project.id}`} className="btn btn-sm btn-ghost">
                                                    Ver →
                                                </Link>
                                            </td>
                                        </tr>
                                    ))
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    );
}
