import { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import axios from 'axios';
import PageHeader from '../../components/PageHeader';

/**
 * 🎨 MÓDULO: DISEÑO WEB - DETALLE
 * 
 * ✅ ESTILO ACTUALIZADO - Usa el diseño premium estándar
 */

export default function WebDesignProjectShow() {
    const { id } = useParams();
    const [project, setProject] = useState<any>(null);
    const [activeTab, setActiveTab] = useState<'overview' | 'pages' | 'deliverables'>('overview');
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const fetchProject = async () => {
            try {
                const res = await axios.get(`/api/web-design/projects/${id}`);
                setProject(res.data);
            } catch (error) {
                console.error(error);
            } finally {
                setLoading(false);
            }
        };
        fetchProject();
    }, [id]);

    if (loading) return <div className="p-8 text-center text-slate-400">Cargando detalles del proyecto...</div>;
    if (!project) return <div className="p-8 text-center text-red-400">Proyecto no encontrado</div>;

    const tabs = [
        { id: 'overview', label: '📊 Resumen', icon: '📊' },
        { id: 'pages', label: '🗺️ Páginas / Sitemap', icon: '🗺️' },
        { id: 'deliverables', label: '📦 Entregables', icon: '📦' },
    ];

    return (
        <div className="animate-fadeIn">
            <Link to="/app/web-design/projects" className="text-sm text-slate-400 hover:text-white mb-4 inline-block">← Volver a proyectos</Link>

            <PageHeader
                title={project.name}
                subtitle={`Estado: ${project.status.toUpperCase()}`}
                actions={
                    <div className="flex gap-2">
                        <button className="btn btn-secondary">✏️ Editar</button>
                    </div>
                }
            />

            {/* Tabs Navigation */}
            <div className="flex border-b border-slate-700 mb-6">
                {tabs.map(tab => (
                    <button
                        key={tab.id}
                        onClick={() => setActiveTab(tab.id as any)}
                        className={`px-6 py-3 font-medium transition-colors border-b-2 ${activeTab === tab.id
                            ? 'border-primary-500 text-white bg-slate-800/50'
                            : 'border-transparent text-slate-400 hover:text-slate-200 hover:bg-slate-800/30'
                            }`}
                    >
                        {tab.label}
                    </button>
                ))}
            </div>

            {/* Tab Content */}
            <div className="min-h-[300px]">
                {activeTab === 'overview' && (
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div className="card">
                            <h3 className="text-lg font-bold mb-4 text-white">Detalles Generales</h3>
                            <div className="space-y-3 text-sm">
                                <div className="flex justify-between border-b border-slate-700 pb-2">
                                    <span className="text-slate-400">Inicio:</span>
                                    <span>{project.start_date || 'N/A'}</span>
                                </div>
                                <div className="flex justify-between border-b border-slate-700 pb-2">
                                    <span className="text-slate-400">Entrega Estimada:</span>
                                    <span>{project.due_date || 'N/A'}</span>
                                </div>
                                <div className="flex justify-between border-b border-slate-700 pb-2">
                                    <span className="text-slate-400">Presupuesto:</span>
                                    <span className="text-emerald-400 font-mono">${project.budget || '0.00'}</span>
                                </div>
                                <div className="flex justify-between pt-2">
                                    <span className="text-slate-400">Staging URL:</span>
                                    {project.staging_url ? (
                                        <a href={project.staging_url} target="_blank" className="text-blue-400 hover:underline">Ver Demo 🔗</a>
                                    ) : <span className="text-slate-500">No configurada</span>}
                                </div>
                            </div>
                        </div>

                        <div className="card">
                            <h3 className="text-lg font-bold mb-4 text-white">Progreso</h3>
                            {/* Placeholder de progreso, se puede conectar a conteo de tareas reales */}
                            <div className="h-4 w-full bg-slate-700 rounded-full overflow-hidden">
                                <div className="h-full bg-blue-500 w-1/3"></div>
                            </div>
                            <p className="mt-2 text-sm text-slate-400">Estimado: 33% completado</p>
                        </div>
                    </div>
                )}

                {activeTab === 'pages' && (
                    <div className="table-container">
                        <div className="table-header flex justify-between">
                            <h3 className="table-title">Estructura del Sitio (Sitemap)</h3>
                            <button className="btn btn-sm btn-primary">➕ Nueva Página</button>
                        </div>
                        <div className="table-wrapper">
                            <table className="table">
                                <thead>
                                    <tr>
                                        <th>Página</th>
                                        <th>Tipo</th>
                                        <th>Prioridad</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {project.pages && project.pages.length > 0 ? project.pages.map((p: any) => (
                                        <tr key={p.id}>
                                            <td className="font-bold text-white">{p.title}</td>
                                            <td><span className="badge badge-default">{p.page_type}</span></td>
                                            <td>{p.priority}</td>
                                            <td><button className="text-slate-400 hover:text-white">✏️</button></td>
                                        </tr>
                                    )) : (
                                        <tr><td colSpan={4} className="text-center p-4 text-slate-500">No hay páginas registradas</td></tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </div>
                )}

                {activeTab === 'deliverables' && (
                    <div className="table-container">
                        <div className="table-header flex justify-between">
                            <h3 className="table-title">Entregables (Mockups, Assets)</h3>
                            <button className="btn btn-sm btn-primary">📤 Subir Entregable</button>
                        </div>
                        <div className="table-wrapper">
                            <table className="table">
                                <thead>
                                    <tr>
                                        <th>Nombre</th>
                                        <th>Tipo</th>
                                        <th>Estado</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {project.deliverables && project.deliverables.length > 0 ? project.deliverables.map((d: any) => (
                                        <tr key={d.id}>
                                            <td className="font-bold text-white">{d.name}</td>
                                            <td>{d.type}</td>
                                            <td><span className="badge badge-purple">{d.status}</span></td>
                                            <td><button className="text-slate-400 hover:text-white">⬇️</button></td>
                                        </tr>
                                    )) : (
                                        <tr><td colSpan={4} className="text-center p-4 text-slate-500">No hay entregables aún</td></tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </div>
                )}
            </div>
        </div>
    );
}
