import { useEffect, useState } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';
import PageHeader from '../../components/PageHeader';

interface DashboardMetrics {
    work_orders: {
        total: number;
        received: number;
        in_progress: number;
        ready: number;
        overdue: number;
    };
    revenue_month: number;
}

export default function WorkshopDashboard() {
    const [stats, setStats] = useState<DashboardMetrics | null>(null);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const fetchStats = async () => {
            try {
                const res = await axios.get('/api/workshop/dashboard');
                setStats(res.data);
            } catch (error) {
                console.error('Error fetching workshop dashboard:', error);
            } finally {
                setLoading(false);
            }
        };
        fetchStats();
    }, []);

    if (loading) {
        return <div className="loading">Cargando...</div>;
    }

    const metricCards = [
        {
            icon: '🔧',
            label: 'Órdenes Activas',
            value: stats?.work_orders.total || 0,
            gradient: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
            glow: 'rgba(102, 126, 234, 0.4)',
        },
        {
            icon: '📥',
            label: 'Recibidas',
            value: stats?.work_orders.received || 0,
            gradient: 'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)',
            glow: 'rgba(240, 147, 251, 0.4)',
        },
        {
            icon: '⚡',
            label: 'En Proceso',
            value: stats?.work_orders.in_progress || 0,
            gradient: 'linear-gradient(135deg, #4facfe 0%, #00f2fe 100%)',
            glow: 'rgba(79, 172, 254, 0.4)',
        },
        {
            icon: '✅',
            label: 'Listas',
            value: stats?.work_orders.ready || 0,
            gradient: 'linear-gradient(135deg, #43e97b 0%, #38f9d7 100%)',
            glow: 'rgba(67, 233, 123, 0.4)',
        },
        {
            icon: '⏰',
            label: 'Atrasadas',
            value: stats?.work_orders.overdue || 0,
            gradient: 'linear-gradient(135deg, #fa709a 0%, #fee140 100%)',
            glow: 'rgba(250, 112, 154, 0.4)',
        },
        {
            icon: '💰',
            label: 'Ingresos del Mes',
            value: `$${stats?.revenue_month?.toLocaleString() || 0}`,
            gradient: 'linear-gradient(135deg, #30cfd0 0%, #330867 100%)',
            glow: 'rgba(48, 207, 208, 0.4)',
        },
    ];

    return (
        <div className="page">
            <PageHeader
                title="🔧 Taller Técnico"
                subtitle="Mantenimiento y reparación de equipos físicos"
                actions={
                    <>
                        <Link to="/app/workshop/assets" className="btn btn-secondary">
                            📦 Ver Equipos
                        </Link>
                        <Link to="/app/workshop/work-orders" className="btn btn-primary">
                            📋 Órdenes de Trabajo
                        </Link>
                    </>
                }
            />

            {/* Metric Cards Grid */}
            <div className="cards-grid">
                {metricCards.map((card, index) => (
                    <div
                        key={index}
                        className="card metric-card"
                        style={{ '--card-glow': card.glow } as React.CSSProperties}
                    >
                        <div
                            className="metric-icon"
                            style={{ background: card.gradient }}
                        >
                            {card.icon}
                        </div>
                        <div className="metric-content">
                            <div className="metric-label">{card.label}</div>
                            <div className="metric-value">{card.value}</div>
                        </div>
                    </div>
                ))}
            </div>

            {/* Quick Actions - Con más espaciado */}
            <div className="card" style={{ marginTop: '2rem' }}>
                <h3 style={{ marginBottom: '1.5rem', fontSize: '1.25rem' }}>Acciones Rápidas</h3>
                <div style={{
                    display: 'grid',
                    gridTemplateColumns: 'repeat(auto-fit, minmax(280px, 1fr))',
                    gap: '1.5rem',
                    padding: '0.5rem 0'
                }}>
                    <Link
                        to="/app/workshop/work-orders?status=received"
                        className="action-card"
                        style={{
                            display: 'flex',
                            alignItems: 'center',
                            gap: '1rem',
                            padding: '1.25rem',
                            background: 'var(--glass-bg)',
                            border: '1px solid var(--border-color)',
                            borderRadius: '12px',
                            textDecoration: 'none',
                            transition: 'all 0.2s ease',
                        }}
                    >
                        <span style={{ fontSize: '2rem' }}>📥</span>
                        <div>
                            <div style={{ fontSize: '1rem', fontWeight: 600, color: 'var(--text-primary)', marginBottom: '0.25rem' }}>
                                Órdenes Recibidas
                            </div>
                            <div style={{ fontSize: '0.875rem', color: 'var(--text-secondary)' }}>
                                Ver equipos recién ingresados
                            </div>
                        </div>
                    </Link>
                    <Link
                        to="/app/workshop/work-orders?status=ready"
                        className="action-card"
                        style={{
                            display: 'flex',
                            alignItems: 'center',
                            gap: '1rem',
                            padding: '1.25rem',
                            background: 'var(--glass-bg)',
                            border: '1px solid var(--border-color)',
                            borderRadius: '12px',
                            textDecoration: 'none',
                            transition: 'all 0.2s ease',
                        }}
                    >
                        <span style={{ fontSize: '2rem' }}>✅</span>
                        <div>
                            <div style={{ fontSize: '1rem', fontWeight: 600, color: 'var(--text-primary)', marginBottom: '0.25rem' }}>
                                Listas para Entrega
                            </div>
                            <div style={{ fontSize: '0.875rem', color: 'var(--text-secondary)' }}>
                                Equipos ya reparados
                            </div>
                        </div>
                    </Link>
                    <Link
                        to="/app/workshop/assets"
                        className="action-card"
                        style={{
                            display: 'flex',
                            alignItems: 'center',
                            gap: '1rem',
                            padding: '1.25rem',
                            background: 'var(--glass-bg)',
                            border: '1px solid var(--border-color)',
                            borderRadius: '12px',
                            textDecoration: 'none',
                            transition: 'all 0.2s ease',
                        }}
                    >
                        <span style={{ fontSize: '2rem' }}>📦</span>
                        <div>
                            <div style={{ fontSize: '1rem', fontWeight: 600, color: 'var(--text-primary)', marginBottom: '0.25rem' }}>
                                Gestión de Activos
                            </div>
                            <div style={{ fontSize: '0.875rem', color: 'var(--text-secondary)' }}>
                                Inventario de equipos
                            </div>
                        </div>
                    </Link>
                </div>
            </div>
        </div>
    );
}
