import React, { useState, useEffect } from 'react';
import { useTheme } from '../../context/ThemeContext';
import { useUser } from '../../context/UserContext';
import axios from 'axios';
import { 
    Microscope, Stethoscope, Syringe, Scan, Heart, 
    FlaskConical, ClipboardList, Activity, Brain, 
    Bone, Eye, Baby, Thermometer, Pill, Hospital
} from 'lucide-react';

interface Categoria {
    id: number;
    nombre: string;
    descripcion: string;
    icono: string;
    color: string;
    estado: string;
    orden: number;
}

// Iconos disponibles de Lucide
const ICONOS_DISPONIBLES: Record<string, any> = {
    'Microscope': Microscope,
    'Stethoscope': Stethoscope,
    'Syringe': Syringe,
    'Scan': Scan,
    'Heart': Heart,
    'FlaskConical': FlaskConical,
    'ClipboardList': ClipboardList,
    'Activity': Activity,
    'Brain': Brain,
    'Bone': Bone,
    'Eye': Eye,
    'Baby': Baby,
    'Thermometer': Thermometer,
    'Pill': Pill,
    'Hospital': Hospital,
};

const NOMBRES_ICONOS = Object.keys(ICONOS_DISPONIBLES);
const COLORES_PRESET = ['#3B82F6', '#10B981', '#F59E0B', '#EF4444', '#8B5CF6', '#EC4899', '#06B6D4', '#84CC16', '#F97316', '#6366F1'];

export default function Categorias() {
    const { theme } = useTheme();
    const { user } = useUser();
    const isDarkMode = theme === 'dark';
    const isAdmin = user?.rol === 'admin';

    const [categorias, setCategorias] = useState<Categoria[]>([]);
    const [loading, setLoading] = useState(true);
    const [buscar, setBuscar] = useState('');
    const [mostrarModal, setMostrarModal] = useState(false);
    const [categoriaEditando, setCategoriaEditando] = useState<Categoria | null>(null);
    const [guardando, setGuardando] = useState(false);
    const [mensaje, setMensaje] = useState({ tipo: '', texto: '' });

    const [formData, setFormData] = useState({
        nombre: '',
        descripcion: '',
        icono: 'Microscope',
        color: '#3B82F6',
        orden: 0,
    });

    useEffect(() => {
        cargarCategorias();
    }, [buscar]);

    const cargarCategorias = async () => {
        setLoading(true);
        try {
            const params: any = {};
            if (buscar) params.buscar = buscar;

            const res = await axios.get('/categorias', { params });
            setCategorias(res.data.data);
        } catch (error) {
            console.error('Error:', error);
        } finally {
            setLoading(false);
        }
    };

    const abrirModalNuevo = () => {
        setCategoriaEditando(null);
        setFormData({ nombre: '', descripcion: '', icono: 'Microscope', color: '#3B82F6', orden: 0 });
        setMensaje({ tipo: '', texto: '' });
        setMostrarModal(true);
    };

    const abrirModalEditar = (categoria: Categoria) => {
        setCategoriaEditando(categoria);
        setFormData({
            nombre: categoria.nombre,
            descripcion: categoria.descripcion || '',
            icono: categoria.icono || 'Microscope',
            color: categoria.color || '#3B82F6',
            orden: categoria.orden || 0,
        });
        setMensaje({ tipo: '', texto: '' });
        setMostrarModal(true);
    };

    const guardarCategoria = async (e: React.FormEvent) => {
        e.preventDefault();
        setGuardando(true);
        setMensaje({ tipo: '', texto: '' });

        try {
            if (categoriaEditando) {
                await axios.put(`/categorias/${categoriaEditando.id}`, formData);
                setMensaje({ tipo: 'success', texto: 'Categoría actualizada' });
            } else {
                await axios.post('/categorias', formData);
                setMensaje({ tipo: 'success', texto: 'Categoría creada' });
            }
            cargarCategorias();
            setTimeout(() => setMostrarModal(false), 1000);
        } catch (error: any) {
            setMensaje({ tipo: 'error', texto: error.response?.data?.message || 'Error al guardar' });
        } finally {
            setGuardando(false);
        }
    };

    const cambiarEstado = async (categoria: Categoria) => {
        const nuevoEstado = categoria.estado === 'activo' ? 'inactivo' : 'activo';
        try {
            await axios.put(`/categorias/${categoria.id}`, { estado: nuevoEstado });
            cargarCategorias();
        } catch (error) { console.error(error); }
    };

    const eliminarCategoria = async (id: number) => {
        if (!confirm('¿Eliminar categoría?')) return;
        try {
            const res = await axios.delete(`/categorias/${id}`);
            if (res.data.success) {
                cargarCategorias();
            } else {
                alert(res.data.message);
            }
        } catch (error: any) {
            alert(error.response?.data?.message || 'Error al eliminar');
        }
    };

    return (
        <div className="space-y-6">
            <div>
                <h1 className={`text-3xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                    📁 Categorías de Estudios
                </h1>
                <p className={`${isDarkMode ? 'text-gray-400' : 'text-gray-600'} mt-1`}>
                    Gestión de categorías para organizar los estudios médicos
                </p>
            </div>

            <div className={`rounded-2xl p-6 ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg`}>
                <div className="flex flex-wrap gap-4 items-center justify-between mb-6">
                    <div className="flex flex-wrap gap-3 items-center">
                        <input type="text" placeholder="Buscar categoría..." value={buscar} onChange={(e) => setBuscar(e.target.value)}
                            className={`px-4 py-2 rounded-xl border-2 outline-none ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} />
                    </div>
                    {isAdmin && (
                        <button onClick={abrirModalNuevo}
                            className="px-6 py-2 rounded-xl font-bold bg-blue-600 text-white">
                            ➕ Nueva Categoría
                        </button>
                    )}
                </div>

                {loading ? (
                    <div className="text-center py-8">
                        <div className={`animate-spin rounded-full h-12 w-12 border-b-2 mx-auto ${isDarkMode ? 'border-white' : 'border-blue-600'}`}></div>
                    </div>
                ) : (
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                        {categorias.length === 0 ? (
                            <div className={`col-span-full text-center p-8 ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>
                                Sin categorías registradas
                            </div>
                        ) : (
                            categorias.map((categoria) => (
                                <div key={categoria.id} className={`rounded-xl p-4 border-2 transition-all hover:scale-105 ${isDarkMode ? 'bg-gray-700/50 border-gray-600' : 'bg-white border-gray-200'}`}>
                                    <div className="flex items-start justify-between mb-3">
                                        <div className="flex items-center gap-3">
                                            <div className="w-12 h-12 rounded-xl flex items-center justify-center" style={{ backgroundColor: categoria.color + '20', border: `2px solid ${categoria.color}` }}>
                                                {(() => {
                                                    const IconComponent = ICONOS_DISPONIBLES[categoria.icono] || Hospital;
                                                    return <IconComponent size={24} color={categoria.color} />;
                                                })()}
                                            </div>
                                            <div>
                                                <h3 className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>{categoria.nombre}</h3>
                                                <p className={`text-xs ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>Orden: {categoria.orden}</p>
                                            </div>
                                        </div>
                                        <button onClick={() => cambiarEstado(categoria)}
                                            className={`px-2 py-1 rounded text-xs ${categoria.estado === 'activo' ? (isDarkMode ? 'bg-green-900 text-green-400' : 'bg-green-100 text-green-700') : (isDarkMode ? 'bg-red-900 text-red-400' : 'bg-red-100 text-red-700')}`}>
                                            {categoria.estado}
                                        </button>
                                    </div>
                                    {categoria.descripcion && (
                                        <p className={`text-sm mb-3 ${isDarkMode ? 'text-gray-300' : 'text-gray-600'}`}>{categoria.descripcion}</p>
                                    )}
                                    {isAdmin && (
                                        <div className="flex justify-end gap-2 pt-3 border-t border-gray-600/20">
                                            <button onClick={() => abrirModalEditar(categoria)} className={`p-2 rounded ${isDarkMode ? 'hover:bg-gray-600' : 'hover:bg-gray-100'}`}>✏️</button>
                                            <button onClick={() => eliminarCategoria(categoria.id)} className={`p-2 rounded ${isDarkMode ? 'hover:bg-gray-600 text-red-400' : 'hover:bg-gray-100 text-red-600'}`}>🗑️</button>
                                        </div>
                                    )}
                                </div>
                            ))
                        )}
                    </div>
                )}
            </div>

            {mostrarModal && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
                    <div className={`rounded-2xl p-6 w-full max-w-md ${isDarkMode ? 'bg-gray-800 border border-gray-700' : 'bg-white border border-gray-200'}`}>
                        <h2 className={`text-2xl font-bold mb-6 ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                            {categoriaEditando ? '✏️ Editar Categoría' : '➕ Nueva Categoría'}
                        </h2>
                        {mensaje.texto && (
                            <div className={`mb-4 p-3 rounded ${mensaje.tipo === 'success' ? (isDarkMode ? 'bg-green-900 text-green-400' : 'bg-green-100 text-green-700') : (isDarkMode ? 'bg-red-900 text-red-400' : 'bg-red-100 text-red-700')}`}>
                                {mensaje.texto}
                            </div>
                        )}
                        <form onSubmit={guardarCategoria} className="space-y-4">
                            <div>
                                <label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Nombre *</label>
                                <input type="text" value={formData.nombre} onChange={(e) => setFormData({...formData, nombre: e.target.value})} required
                                    className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} />
                            </div>
                            <div>
                                <label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Descripción</label>
                                <textarea value={formData.descripcion} onChange={(e) => setFormData({...formData, descripcion: e.target.value})} rows={2}
                                    className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} />
                            </div>
                            <div className="grid grid-cols-2 gap-4">
                                <div>
                                    <label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Icono</label>
                                    <select value={formData.icono} onChange={(e) => setFormData({...formData, icono: e.target.value})}
                                        className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`}>
                                        {NOMBRES_ICONOS.map((iconName: string) => <option key={iconName} value={iconName}>{iconName}</option>)}
                                    </select>
                                </div>
                                <div>
                                    <label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Orden</label>
                                    <input type="number" value={formData.orden} onChange={(e) => setFormData({...formData, orden: parseInt(e.target.value) || 0})}
                                        className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} />
                                </div>
                            </div>
                            <div>
                                <label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Color</label>
                                <div className="flex flex-wrap gap-2">
                                    {COLORES_PRESET.map(color => (
                                        <button key={color} type="button" onClick={() => setFormData({...formData, color})}
                                            className={`w-8 h-8 rounded-full border-2 ${formData.color === color ? 'border-white ring-2 ring-gray-400' : 'border-transparent'}`}
                                            style={{ backgroundColor: color }} />
                                    ))}
                                </div>
                            </div>
                            <div className="flex gap-3 pt-4">
                                <button type="button" onClick={() => setMostrarModal(false)} className={`flex-1 p-2 rounded ${isDarkMode ? 'bg-gray-700 text-gray-300' : 'bg-gray-200 text-gray-700'}`}>Cancelar</button>
                                <button type="submit" disabled={guardando} className={`flex-1 p-2 rounded text-white bg-blue-600 disabled:opacity-50`}>
                                    {guardando ? '...' : categoriaEditando ? 'Actualizar' : 'Crear'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
