import React, { useState, useEffect } from 'react';
import { useTheme } from '../../context/ThemeContext';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import { Calendar, Clock, FileText, User, CreditCard, CheckCircle, XCircle, Filter, Plus } from 'lucide-react';

interface Paciente {
    id: number;
    nombre: string;
    apellido: string;
    numero_documento: string;
}

interface Medico {
    id: number;
    nombre: string;
    apellido: string;
    rol: string;
}

interface Cita {
    id: number;
    paciente: {
        id: number;
        nombre: string;
        apellido: string;
    };
    // examen?: Examen | null; // Legacy no usa examen en cita
    fecha_cita: string;
    creada_por: {
        nombre: string;
        apellido: string;
    } | null;
    estado: 'programada' | 'confirmada' | 'en_curso' | 'completada' | 'cancelada' | 'no_asistio';
    pagado: boolean; // Legacy siempre false/undefined
    fecha_creacion: string;
    tipo_cita: string;
    motivo_consulta: string;
}

export default function Citas() {
    const { theme } = useTheme();
    const isDarkMode = theme === 'dark';
    const navigate = useNavigate();

    // Estados de datos
    const [citas, setCitas] = useState<Cita[]>([]);
    const [pacientes, setPacientes] = useState<Paciente[]>([]);
    const [medicos, setMedicos] = useState<Medico[]>([]);

    // Estados de UI
    const [loading, setLoading] = useState(true);
    const [loadingCatalogos, setLoadingCatalogos] = useState(false);
    const [errorCatalogos, setErrorCatalogos] = useState<string | null>(null);
    const [showModal, setShowModal] = useState(false);
    const [filtroEstado, setFiltroEstado] = useState('');
    const [submitting, setSubmitting] = useState(false);

    // Formulario
    const [formData, setFormData] = useState({
        paciente_id: '',
        medico_id: '',
        fecha_cita: '',
        hora_cita: '',
        tipo_cita: 'consulta_general',
        motivo_consulta: '',
        duracion_minutos: 30,
        observaciones: ''
    });

    useEffect(() => {
        fetchCitas();
        loadCatalogos();
    }, []);

    const fetchCitas = async () => {
        try {
            const token = localStorage.getItem('token');
            if (!token) return;
            const res = await axios.get('/citas', {
                headers: { 'Authorization': `Bearer ${token}` }
            });
            if (res.data.success) {
                // Ordenar por fecha_cita descendente si no viene ordenado
                const sorted = res.data.data.sort((a: Cita, b: Cita) =>
                    new Date(b.fecha_cita).getTime() - new Date(a.fecha_cita).getTime()
                );
                setCitas(sorted);
            }
        } catch (error) {
            console.error('Error al cargar citas:', error);
        } finally {
            setLoading(false);
        }
    };

    const loadCatalogos = async () => {
        setLoadingCatalogos(true);
        setErrorCatalogos(null);
        try {
            const token = localStorage.getItem('token');
            if (!token) {
                console.error("No token found for catalogs");
                setErrorCatalogos("No se encontró token de autenticación.");
                return;
            }

            console.log("Cargando catálogos...");

            // Cargar Pacientes
            try {
                const resPacientes = await axios.get('/pacientes?per_page=100', {
                    headers: { 'Authorization': `Bearer ${token}` }
                });
                console.log("Respuesta Pacientes:", resPacientes.status, resPacientes.data);

                const pData = resPacientes.data;

                if (pData && pData.success === true && Array.isArray(pData.data)) {
                    setPacientes(pData.data);
                    console.log("Pacientes cargados:", pData.data.length);
                } else if (pData && Array.isArray(pData)) {
                    // Fallback: quizás la API devuelve un array directo
                    setPacientes(pData);
                    console.log("Pacientes (array directo):", pData.length);
                } else if (typeof pData === 'string' && pData.includes('<!DOCTYPE')) {
                    // La respuesta es HTML (probablemente redirect de login)
                    setErrorCatalogos("Sesión expirada. Por favor, cierra sesión y vuelve a ingresar.");
                } else {
                    console.warn("Formato inesperado:", JSON.stringify(pData).substring(0, 200));
                    setErrorCatalogos(`Error: formato de respuesta inesperado. Status: ${resPacientes.status}`);
                }
            } catch (err: any) {
                console.error("Error cargando pacientes:", err);
                if (err?.response?.status === 401) {
                    setErrorCatalogos("Sesión expirada. Por favor, cierra sesión y vuelve a ingresar.");
                } else if (err?.response?.status === 302 || err?.response?.status === 200) {
                    // Verificar si es HTML (redirect)
                    const responseText = err.response?.data;
                    if (typeof responseText === 'string' && responseText.includes('<!DOCTYPE')) {
                        setErrorCatalogos("Sesión expirada. Por favor, cierra sesión y vuelve a ingresar.");
                    } else {
                        setErrorCatalogos("Error de autenticación: el token guardado no es válido en este puerto.");
                    }
                } else {
                    setErrorCatalogos(`Error al cargar pacientes: ${err?.message || 'desconocido'}`);
                }
            }

            // Cargar Médicos
            try {
                const resUsuarios = await axios.get('/usuarios?per_page=100', {
                    headers: { 'Authorization': `Bearer ${token}` }
                });
                console.log("Respuesta Usuarios:", resUsuarios.status, resUsuarios.data);
                if (resUsuarios.data.data && Array.isArray(resUsuarios.data.data)) {
                    // Filtrar solo medicos
                    const listaMedicos = resUsuarios.data.data.filter((u: Medico) => u.rol === 'medico' || u.rol === 'admin');
                    setMedicos(listaMedicos);
                    console.log("Médicos cargados:", listaMedicos.length);
                } else {
                    console.warn("Formato de respuesta usuarios inválido", resUsuarios.data);
                }
            } catch (err) {
                console.error("Error cargando médicos:", err);
                // If patients loaded, but doctors failed, we might still want to proceed.
                // If both fail, the first error (patients) will likely be shown.
            }

        } catch (error) {
            console.error("Error general cargando catálogos", error);
            setErrorCatalogos("Error de conexión al cargar datos");
        } finally {
            setLoadingCatalogos(false);
        }
    };

    const handleCreateCita = async (e: React.FormEvent) => {
        e.preventDefault();
        setSubmitting(true);
        try {
            const token = localStorage.getItem('token');
            // Combinar fecha y hora
            const fechaFinal = `${formData.fecha_cita} ${formData.hora_cita}:00`;

            const payload = {
                paciente_id: formData.paciente_id,
                medico_id: formData.medico_id,
                fecha_cita: fechaFinal,
                duracion_minutos: formData.duracion_minutos,
                tipo_cita: formData.tipo_cita,
                motivo_consulta: formData.motivo_consulta,
                observaciones: formData.observaciones,
                pagado: false // Legacy
            };

            const res = await axios.post('/citas', payload, {
                headers: { 'Authorization': `Bearer ${token}` }
            });

            if (res.data.success) {
                const conf = window.confirm('Cita creada exitosamente.\n¿Desea ir a preparar la Historia Clínica para este paciente ahora?');
                if (conf) {
                    navigate('/historias-clinicas');
                }

                setShowModal(false);
                fetchCitas();
                setFormData({
                    paciente_id: '',
                    medico_id: '',
                    fecha_cita: '',
                    hora_cita: '',
                    tipo_cita: 'consulta_general',
                    motivo_consulta: '',
                    duracion_minutos: 30,
                    observaciones: ''
                });
            }

        } catch (error: any) {
            console.error(error);
            alert(error.response?.data?.message || 'Error al crear la cita');
        } finally {
            setSubmitting(false);
        }
    };

    // 🎨 Colores (Mismo código de antes)
    const getEstadoColor = (estado: string) => {
        const colors: Record<string, { light: string; dark: string; icon: string }> = {
            programada: { light: 'bg-yellow-100 text-yellow-800 border-yellow-300', dark: 'bg-yellow-900/40 text-yellow-300 border-yellow-700', icon: '🟡' },
            confirmada: { light: 'bg-green-100 text-green-800 border-green-300', dark: 'bg-green-900/40 text-green-300 border-green-700', icon: '🟢' },
            en_curso: { light: 'bg-blue-100 text-blue-800 border-blue-300', dark: 'bg-blue-900/40 text-blue-300 border-blue-700', icon: '🔵' },
            completada: { light: 'bg-purple-100 text-purple-800 border-purple-300', dark: 'bg-purple-900/40 text-purple-300 border-purple-700', icon: '🟣' },
            cancelada: { light: 'bg-red-100 text-red-800 border-red-300', dark: 'bg-red-900/40 text-red-300 border-red-700', icon: '🔴' },
            no_asistio: { light: 'bg-gray-100 text-gray-800 border-gray-300', dark: 'bg-gray-700 text-gray-400 border-gray-600', icon: '⚫' }
        };
        const color = colors[estado] || colors.programada;
        return isDarkMode ? color.dark : color.light;
    };

    const getEstadoLabel = (estado: string) => estado.charAt(0).toUpperCase() + estado.slice(1).replace('_', ' ');

    const filteredCitas = citas.filter(cita => filtroEstado ? cita.estado === filtroEstado : true);

    const formatFecha = (dateString: string) => {
        if (!dateString) return '-';
        const date = new Date(dateString);
        return new Intl.DateTimeFormat('es-BO', { day: '2-digit', month: 'short', year: 'numeric' }).format(date);
    };

    const formatHora = (dateString: string) => {
        if (!dateString) return '-';
        const date = new Date(dateString);
        return new Intl.DateTimeFormat('es-BO', { hour: '2-digit', minute: '2-digit', hour12: true }).format(date);
    };

    return (
        <div className="space-y-6">
            {/* Header Updated with Button */}
            <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                <div>
                    <h1 className={`text-2xl md:text-3xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'} flex items-center gap-2`}>
                        <Calendar className="w-7 h-7 md:w-8 md:h-8 text-blue-500" />
                        Citas Médicas
                    </h1>
                    <p className={`${isDarkMode ? 'text-gray-400' : 'text-gray-600'} mt-1 text-sm md:text-base`}>
                        Gestión de agenda y citas
                    </p>
                </div>
                <div className="flex items-center gap-4">
                    <div className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-600'} hidden md:block`}>
                        Total: <span className="font-bold text-blue-500">{filteredCitas.length}</span>
                    </div>
                    <button
                        onClick={() => setShowModal(true)}
                        className="px-4 py-2 bg-gradient-to-r from-blue-500 to-cyan-500 text-white rounded-xl font-semibold shadow-lg hover:shadow-xl transform hover:scale-105 transition-all flex items-center gap-2"
                    >
                        <Plus className="w-5 h-5" />
                        Nueva Cita
                    </button>
                </div>
            </div>

            {/* Filters */}
            <div className={`p-4 rounded-xl ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg`}>
                <div className="flex items-center gap-2 mb-3">
                    <Filter className={`w-4 h-4 ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`} />
                    <span className={`text-sm font-semibold ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                        Filtrar por Estado:
                    </span>
                </div>
                <select
                    value={filtroEstado}
                    onChange={(e) => setFiltroEstado(e.target.value)}
                    className={`w-full md:w-1/3 px-4 py-2 rounded-xl border-2 outline-none transition-all ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200 text-gray-700'}`}
                >
                    <option value="">Todos los estados</option>
                    <option value="programada">🟡 Programada</option>
                    <option value="confirmada">🟢 Confirmada</option>
                    <option value="en_curso">🔵 En Curso</option>
                    <option value="completada">🟣 Completada</option>
                    <option value="cancelada">🔴 Cancelada</option>
                    <option value="no_asistio">⚫ No Asistió</option>
                </select>
            </div>

            {/* List View */}
            <div className={`rounded-2xl overflow-hidden ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg`}>
                {loading ? (
                    <div className="p-12 text-center"><div className="animate-spin rounded-full h-10 w-10 border-b-2 border-blue-500 mx-auto"></div></div>
                ) : filteredCitas.length === 0 ? (
                    <div className={`p-12 text-center ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>No hay citas registradas.</div>
                ) : (
                    <div className="overflow-x-auto">
                        <table className="w-full">
                            <thead className={`${isDarkMode ? 'bg-gray-900/50 text-gray-400' : 'bg-gray-50 text-gray-600'}`}>
                                <tr>
                                    <th className="px-6 py-4 text-left">Paciente</th>
                                    <th className="px-6 py-4 text-left">Motivo / Tipo</th>
                                    <th className="px-6 py-4 text-left">Fecha y Hora</th>
                                    <th className="px-6 py-4 text-left">Médico</th>
                                    <th className="px-6 py-4 text-center">Estado</th>
                                    <th className="px-6 py-4 text-center">Acciones</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y dark:divide-gray-700">
                                {filteredCitas.map((cita) => (
                                    <tr key={cita.id} className={`transition-colors ${isDarkMode ? 'hover:bg-gray-700/30' : 'hover:bg-gray-50'}`}>
                                        <td className="px-6 py-4">
                                            <div className={`font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>{cita.paciente.nombre} {cita.paciente.apellido}</div>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className={`text-sm ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>{cita.tipo_cita}</div>
                                            <div className="text-xs text-gray-500">{cita.motivo_consulta}</div>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className={`text-sm font-mono ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                                {formatFecha(cita.fecha_cita)} <span className="text-gray-400">|</span> {formatHora(cita.fecha_cita)}
                                            </div>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className="text-sm">{cita.creada_por?.nombre} {cita.creada_por?.apellido}</div>
                                        </td>
                                        <td className="px-6 py-4 text-center">
                                            <span className={`px-3 py-1 text-xs font-bold rounded-full border ${getEstadoColor(cita.estado)}`}>
                                                {getEstadoLabel(cita.estado)}
                                            </span>
                                        </td>
                                        <td className="px-6 py-4 text-center">
                                            {['programada', 'confirmada', 'en_curso'].includes(cita.estado) && (
                                                <button
                                                    onClick={() => navigate('/historias-clinicas')}
                                                    className="px-3 py-1.5 text-xs font-bold text-white bg-blue-600 hover:bg-blue-700 rounded-lg shadow-sm transition-colors"
                                                >
                                                    🩺 Iniciar Consulta
                                                </button>
                                            )}
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                )}
            </div>

            {/* Modal Nueva Cita */}
            {showModal && (
                <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/60 backdrop-blur-sm p-4">
                    <div className={`w-full max-w-lg rounded-2xl shadow-2xl ${isDarkMode ? 'bg-gray-800 border border-gray-700' : 'bg-white'}`}>
                        <div className="p-6 border-b dark:border-gray-700">
                            <h2 className={`text-xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>Agendar Nueva Cita</h2>
                        </div>

                        {loadingCatalogos ? (
                            <div className="p-12 flex flex-col items-center justify-center">
                                <div className="animate-spin rounded-full h-10 w-10 border-b-2 border-blue-500 mb-4"></div>
                                <p className={isDarkMode ? 'text-gray-300' : 'text-gray-600'}>Cargando lista de pacientes y médicos...</p>
                            </div>
                        ) : errorCatalogos ? (
                            <div className="p-8 text-center">
                                <div className="text-red-500 mb-4">
                                    <XCircle className="w-12 h-12 mx-auto mb-2" />
                                    <p className="font-semibold">{errorCatalogos}</p>
                                </div>
                                <button
                                    onClick={loadCatalogos}
                                    className="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors"
                                >
                                    Reintentar Cargar Datos
                                </button>
                                <button
                                    onClick={() => setShowModal(false)}
                                    className="block mx-auto mt-4 text-sm text-gray-500 hover:underline"
                                >
                                    Cerrar
                                </button>
                            </div>
                        ) : (
                            <form onSubmit={handleCreateCita} className="p-6 space-y-4">
                                <div>
                                    <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Paciente <span className="text-xs font-normal text-gray-500">({pacientes.length} disponibles)</span>
                                    </label>
                                    <select
                                        required
                                        className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                        value={formData.paciente_id}
                                        onChange={e => setFormData({ ...formData, paciente_id: e.target.value })}
                                    >
                                        <option value="">Seleccione un paciente...</option>
                                        {pacientes.map(p => (
                                            <option key={p.id} value={p.id}>{p.nombre} {p.apellido} ({p.numero_documento})</option>
                                        ))}
                                    </select>
                                    {pacientes.length === 0 && (
                                        <p className="text-xs text-orange-500 mt-1">
                                            No se encontraron pacientes. Revise la consola (F12) para más detalles.
                                        </p>
                                    )}
                                </div>

                                <div>
                                    <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Médico</label>
                                    <select
                                        required
                                        className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                        value={formData.medico_id}
                                        onChange={e => setFormData({ ...formData, medico_id: e.target.value })}
                                    >
                                        <option value="">Seleccione un médico...</option>
                                        {medicos.map(m => (
                                            <option key={m.id} value={m.id}>{m.nombre} {m.apellido}</option>
                                        ))}
                                    </select>
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Fecha</label>
                                        <input
                                            type="date"
                                            required
                                            className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                            value={formData.fecha_cita}
                                            onChange={e => setFormData({ ...formData, fecha_cita: e.target.value })}
                                        />
                                    </div>
                                    <div>
                                        <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Hora</label>
                                        <input
                                            type="time"
                                            required
                                            className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                            value={formData.hora_cita}
                                            onChange={e => setFormData({ ...formData, hora_cita: e.target.value })}
                                        />
                                    </div>
                                </div>

                                <div>
                                    <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Tipo de Cita</label>
                                    <select
                                        className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                        value={formData.tipo_cita}
                                        onChange={e => setFormData({ ...formData, tipo_cita: e.target.value })}
                                    >
                                        <option value="consulta_general">Consulta General</option>
                                        <option value="consulta_especializada">Consulta Especializada</option>
                                        <option value="control">Control</option>
                                        <option value="emergencia">Emergencia</option>
                                    </select>
                                </div>

                                <div>
                                    <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Motivo</label>
                                    <textarea
                                        required
                                        rows={2}
                                        className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                        value={formData.motivo_consulta}
                                        onChange={e => setFormData({ ...formData, motivo_consulta: e.target.value })}
                                        placeholder="Motivo de la consulta..."
                                    ></textarea>
                                </div>

                                <div className="flex justify-end gap-3 mt-6">
                                    <button
                                        type="button"
                                        onClick={() => setShowModal(false)}
                                        className={`px-4 py-2 rounded-lg font-medium ${isDarkMode ? 'text-gray-300 hover:bg-gray-700' : 'text-gray-600 hover:bg-gray-100'}`}
                                    >
                                        Cancelar
                                    </button>
                                    <button
                                        type="submit"
                                        disabled={submitting}
                                        className="px-6 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg font-semibold shadow-lg transition-colors disabled:opacity-50"
                                    >
                                        {submitting ? 'Agendando...' : 'Agendar Cita'}
                                    </button>
                                </div>
                            </form>
                        )}
                    </div>
                </div>
            )}
        </div>
    );
}
