import React, { useEffect, useState } from 'react';
import { Link } from 'react-router-dom';
import { useTheme } from '../../context/ThemeContext';
import { useUser } from '../../context/UserContext';

export default function Dashboard() {
    const { isDarkMode } = useTheme();
    const { user } = useUser();
    const [stats, setStats] = useState({
        pacientesActivos: 247,
        citasHoy: 12,
        citasPendientes: 8,
        historiasMes: 156
    });

    const statsCards = [
        {
            title: 'Pacientes Activos',
            value: stats.pacientesActivos,
            icon: '👥',
            color: 'from-green-500 to-emerald-500',
            bgLight: 'bg-green-50',
            bgDark: 'bg-green-900/20',
            change: '+12%',
            changePositive: true
        },
        {
            title: 'Citas de Hoy',
            value: stats.citasHoy,
            icon: '📅',
            color: 'from-blue-500 to-cyan-500',
            bgLight: 'bg-blue-50',
            bgDark: 'bg-blue-900/20',
            change: '+5',
            changePositive: true
        },
        {
            title: 'Citas Pendientes',
            value: stats.citasPendientes,
            icon: '⏱️',
            color: 'from-orange-500 to-red-500',
            bgLight: 'bg-orange-50',
            bgDark: 'bg-orange-900/20',
            change: '-3',
            changePositive: true
        },
        {
            title: 'Historias este Mes',
            value: stats.historiasMes,
            icon: '📋',
            color: 'from-purple-500 to-pink-500',
            bgLight: 'bg-purple-50',
            bgDark: 'bg-purple-900/20',
            change: '+23%',
            changePositive: true
        }
    ];

    const quickActions = [
        { title: 'Nueva Cita', icon: '➕', path: '/citas', color: 'from-blue-500 to-cyan-500' },
        { title: 'Nuevo Paciente', icon: '👤', path: '/pacientes', color: 'from-green-500 to-emerald-500' },
        { title: 'Nueva Receta', icon: '💊', path: '/recetas', color: 'from-purple-500 to-pink-500' },
        { title: 'Ver Exámenes', icon: '🔬', path: '/examenes', color: 'from-orange-500 to-red-500' }
    ];

    const recentAppointments = [
        { id: 1, patient: 'Carlos Fernández', time: '09:00 AM', type: 'Consulta General', status: 'confirmada' },
        { id: 2, patient: 'Ana Sánchez', time: '10:30 AM', type: 'Control', status: 'en_curso' },
        { id: 3, patient: 'Luis Ramírez', time: '11:00 AM', type: 'Seguimiento', status: 'programada' },
        { id: 4, patient: 'María González', time: '02:00 PM', type: 'Emergencia', status: 'programada' }
    ];

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'confirmada': return 'bg-green-100 text-green-800 border-green-300';
            case 'en_curso': return 'bg-blue-100 text-blue-800 border-blue-300';
            case 'programada': return 'bg-yellow-100 text-yellow-800 border-yellow-300';
            default: return 'bg-gray-100 text-gray-800 border-gray-300';
        }
    };

    return (
        <div className="space-y-8">
            {/* Welcome Header */}
            <div className={`rounded-3xl p-8 ${isDarkMode ? 'bg-gradient-to-br from-blue-900/40 to-cyan-900/40 border border-blue-700/30' : 'bg-gradient-to-br from-blue-500 to-cyan-400'} shadow-2xl`}>
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-4xl font-bold text-white mb-2">
                            ¡Bienvenido, {user?.nombre || 'Doctor'}! 👋
                        </h1>
                        <p className="text-white/90 text-lg">
                            Hoy es {new Date().toLocaleDateString('es-ES', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })}
                        </p>
                    </div>
                    <div className="hidden md:block">
                        <div className="w-32 h-32 bg-white/20 backdrop-blur-md rounded-3xl flex items-center justify-center text-6xl shadow-xl">
                            🏥
                        </div>
                    </div>
                </div>
            </div>

            {/* Stats Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                {statsCards.map((stat, index) => (
                    <div
                        key={index}
                        className={`rounded-2xl p-6 ${isDarkMode ? stat.bgDark : stat.bgLight} ${isDarkMode ? 'border border-gray-700/50' : 'border border-gray-200'} shadow-lg hover:shadow-xl transition-all hover:scale-105 cursor-pointer`}
                    >
                        <div className="flex items-center justify-between mb-4">
                            <div className={`w-14 h-14 bg-gradient-to-br ${stat.color} rounded-xl flex items-center justify-center text-3xl shadow-lg`}>
                                {stat.icon}
                            </div>
                            <span className={`text-sm font-semibold ${stat.changePositive ? 'text-green-600' : 'text-red-600'}`}>
                                {stat.change}
                            </span>
                        </div>
                        <h3 className={`text-sm font-semibold ${isDarkMode ? 'text-gray-400' : 'text-gray-600'} mb-1`}>
                            {stat.title}
                        </h3>
                        <p className={`text-3xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                            {stat.value}
                        </p>
                    </div>
                ))}
            </div>

            {/* Quick Actions */}
            <div>
                <h2 className={`text-2xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'} mb-4`}>
                    ⚡ Acciones Rápidas
                </h2>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    {quickActions.map((action, index) => (
                        <Link
                            key={index}
                            to={action.path}
                            className={`rounded-2xl p-6 bg-gradient-to-br ${action.color} text-white shadow-lg hover:shadow-2xl transition-all hover:scale-105 text-center`}
                        >
                            <div className="text-5xl mb-3">{action.icon}</div>
                            <p className="font-bold text-lg">{action.title}</p>
                        </Link>
                    ))}
                </div>
            </div>

            {/* Recent Appointments */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                {/* Appointments List */}
                <div className={`rounded-2xl p-6 ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg`}>
                    <h2 className={`text-xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'} mb-4`}>
                        📅 Citas de Hoy
                    </h2>
                    <div className="space-y-3">
                        {recentAppointments.map((appointment) => (
                            <div
                                key={appointment.id}
                                className={`p-4 rounded-xl ${isDarkMode ? 'bg-gray-700/50' : 'bg-gray-50'} hover:shadow-md transition-all cursor-pointer`}
                            >
                                <div className="flex items-center justify-between">
                                    <div>
                                        <p className={`font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                                            {appointment.patient}
                                        </p>
                                        <p className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                                            {appointment.time} • {appointment.type}
                                        </p>
                                    </div>
                                    <span className={`px-3 py-1 rounded-lg text-xs font-semibold border ${getStatusColor(appointment.status)}`}>
                                        {appointment.status}
                                    </span>
                                </div>
                            </div>
                        ))}
                    </div>
                    <Link
                        to="/citas"
                        className="block mt-4 text-center text-blue-600 hover:text-blue-700 font-semibold"
                    >
                        Ver todas las citas →
                    </Link>
                </div>

                {/* Activity Summary */}
                <div className={`rounded-2xl p-6 ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg`}>
                    <h2 className={`text-xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'} mb-4`}>
                        📊 Resumen de Actividad
                    </h2>
                    <div className="space-y-4">
                        <div className={`p-4 rounded-xl ${isDarkMode ? 'bg-blue-900/20' : 'bg-blue-50'}`}>
                            <div className="flex items-center gap-3">
                                <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-cyan-500 rounded-lg flex items-center justify-center text-white">
                                    ✅
                                </div>
                                <div>
                                    <p className={`font-semibold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                                        8 Citas Completadas
                                    </p>
                                    <p className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                                        Esta semana
                                    </p>
                                </div>
                            </div>
                        </div>

                        <div className={`p-4 rounded-xl ${isDarkMode ? 'bg-green-900/20' : 'bg-green-50'}`}>
                            <div className="flex items-center gap-3">
                                <div className="w-10 h-10 bg-gradient-to-br from-green-500 to-emerald-500 rounded-lg flex items-center justify-center text-white">
                                    👥
                                </div>
                                <div>
                                    <p className={`font-semibold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                                        15 Nuevos Pacientes
                                    </p>
                                    <p className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                                        Este mes
                                    </p>
                                </div>
                            </div>
                        </div>

                        <div className={`p-4 rounded-xl ${isDarkMode ? 'bg-purple-900/20' : 'bg-purple-50'}`}>
                            <div className="flex items-center gap-3">
                                <div className="w-10 h-10 bg-gradient-to-br from-purple-500 to-pink-500 rounded-lg flex items-center justify-center text-white">
                                    💊
                                </div>
                                <div>
                                    <p className={`font-semibold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                                        23 Recetas Generadas
                                    </p>
                                    <p className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                                        Este mes
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}
