import React, { useState, useEffect } from 'react';
import { useLocation } from 'react-router-dom';
import axios from 'axios';
import { useTheme } from '../../context/ThemeContext';
import { useUser } from '../../context/UserContext';
import { HistoriaClinica, SoapNotes, ExamenFisico, Cie10 } from '../../types/HistoriaClinica';
import Cie10Select from '../../components/medical/forms/Cie10Select';
import SoapForm from '../../components/medical/forms/SoapForm';
import FileUploader from '../../components/medical/forms/FileUploader';
import RecetaForm, { MedicamentoItem } from '../../components/medical/forms/RecetaForm';
import SignosVitalesForm from '../../components/medical/forms/SignosVitalesForm';

// Tipos auxiliares
interface Paciente {
    id: number;
    nombre: string;
    apellido: string;
    numero_documento: string;
    foto_paciente?: string;
    fecha_nacimiento?: string;
    grupo_sanguineo?: string;
    alergias?: string;
    edad?: number;
    contacto_emergencia?: string;
    telefono_emergencia?: string;
}

interface CitaPendiente {
    id: number;
    fecha_cita: string;
    motivo_consulta: string;
}

export default function HistoriasClinicas() {
    const { isDarkMode } = useTheme();
    const { user } = useUser();

    // Estados principales
    const [searchQuery, setSearchQuery] = useState('');
    const [pacientes, setPacientes] = useState<Paciente[]>([]);
    const [selectedPaciente, setSelectedPaciente] = useState<Paciente | null>(null);
    const [historial, setHistorial] = useState<HistoriaClinica[]>([]);
    const [citasPendientes, setCitasPendientes] = useState<CitaPendiente[]>([]);

    // Estado de creación
    const [isCreating, setIsCreating] = useState(false);
    const [selectedCitaId, setSelectedCitaId] = useState<number | null>(null);
    const [loading, setLoading] = useState(false);

    // --- ESTADOS DEL FORMULARIO ---
    const [soap, setSoap] = useState<SoapNotes>({ subjetivo: '', objetivo: '', analisis: '', plan: '' });
    const [examenFisico, setExamenFisico] = useState<ExamenFisico[]>([]);
    const [cie10, setCie10] = useState<Cie10 | null>(null);
    const [diagnosticoTipo, setDiagnosticoTipo] = useState<'presuntivo' | 'definitivo'>('presuntivo');
    const [archivos, setArchivos] = useState<File[]>([]);
    const [motivoConsulta, setMotivoConsulta] = useState('');
    const [signosVitales, setSignosVitales] = useState({
        presion_arterial: '', frecuencia_cardiaca: '', frecuencia_respiratoria: '', temperatura: '', saturacion_oxigeno: '', peso: '', altura: ''
    });
    const [antecedentes, setAntecedentes] = useState<any[]>([]);

    // DIAGNOSTICOS
    const [diagnosticosList, setDiagnosticosList] = useState<{ cie10: Cie10; tipo: 'presuntivo' | 'definitivo' }[]>([]);

    // RECETA Y PLAN ESTADO
    const [medicamentos, setMedicamentos] = useState<MedicamentoItem[]>([]);
    const [recetaInstrucciones, setRecetaInstrucciones] = useState('');
    const [ordenesLab, setOrdenesLab] = useState<{ id: string, name: string, checked: boolean }[]>([
        { id: 'bh', name: 'Biometría Hemática', checked: false },
        { id: 'qs', name: 'Química Sanguínea', checked: false },
        { id: 'ego', name: 'Examen General de Orina', checked: false },
        { id: 'rx_torax', name: 'Radiografía de Tórax', checked: false }
    ]);
    const [indicacionesGenerales, setIndicacionesGenerales] = useState('');
    const [proximaCita, setProximaCita] = useState('');

    // DEBOUNCE SEARCH
    useEffect(() => {
        if (searchQuery.length > 2) {
            const timer = setTimeout(() => {
                axios.get(`/pacientes?buscar=${searchQuery}`)
                    .then(res => setPacientes(res.data.data || res.data))
                    .catch(console.error);
            }, 300);
            return () => clearTimeout(timer);
        }
    }, [searchQuery]);

    // CHECK URL STATE PARA SELECCION EXTERNA
    const location = useLocation();
    useEffect(() => {
        if (location.state?.pacienteId) {
            axios.get(`/pacientes/${location.state.pacienteId}`)
                .then(res => {
                    setSelectedPaciente(res.data);
                    // Actualizar dropdown search mock
                    setSearchQuery(res.data.numero_documento);
                })
                .catch(console.error);
        }
    }, [location.state]);

    // CARGAR HISTORIAL AL SELECCIONAR PACIENTE
    useEffect(() => {
        if (selectedPaciente) {
            setLoading(true);
            // 1. Cargar historial
            axios.get(`/historias-clinicas?paciente_id=${selectedPaciente.id}`)
                .then(res => setHistorial(res.data.data))
                .catch(console.error);

            // 2. Cargar citas pendientes (para poder atender)
            // Esto requeriría un endpoint específico, usaré uno genérico de citas por ahora
            axios.get(`/citas?paciente_id=${selectedPaciente.id}&estado=programada`)
                .then(res => setCitasPendientes(res.data.data || [])) // Asumiendo estructura paginada
                .catch(console.error)
                .finally(() => setLoading(false));

            // Cargar antecedentes
            axios.get(`/pacientes/${selectedPaciente.id}/antecedentes`)
                .then(res => setAntecedentes(res.data))
                .catch(console.error);

            setIsCreating(false);
        }
    }, [selectedPaciente]);

    const handleStartConsultation = (cita: CitaPendiente) => {
        setSelectedCitaId(cita.id);
        setMotivoConsulta(cita.motivo_consulta);
        setIsCreating(true);
        // Reset form
        setSoap({ subjetivo: '', objetivo: '', analisis: '', plan: '' });
        setExamenFisico([]);
        setCie10(null);
        setDiagnosticosList([]);
        setArchivos([]);
        setMedicamentos([]);
        setRecetaInstrucciones('');
        setIndicacionesGenerales('');
        setProximaCita('');
        setOrdenesLab(prev => prev.map(o => ({ ...o, checked: false })));
        setSignosVitales({
            presion_arterial: '', frecuencia_cardiaca: '', frecuencia_respiratoria: '', temperatura: '', saturacion_oxigeno: '', peso: '', altura: ''
        });
    };

    const handleSaveHistoria = async () => {
        const mainDiag = cie10 || diagnosticosList[0]?.cie10;
        const mainDiagTipo = cie10 ? diagnosticoTipo : diagnosticosList[0]?.tipo;

        if (!selectedPaciente || !selectedCitaId || !mainDiag) {
            alert('Faltan datos obligatorios (Paciente, Cita o Diagnóstico)');
            return;
        }

        setLoading(true);
        try {
            const formData = new FormData();
            formData.append('paciente_id', selectedPaciente.id.toString());
            formData.append('medico_id', user?.id.toString() || '1');
            formData.append('cita_id', selectedCitaId.toString());
            formData.append('motivo_consulta', motivoConsulta);

            formData.append('cie10_id', mainDiag.id.toString());
            formData.append('tipo_diagnostico', mainDiagTipo);

            // Agregar Signos Vitales
            formData.append('presion_arterial', signosVitales.presion_arterial);
            formData.append('peso', signosVitales.peso);
            formData.append('altura', signosVitales.altura);
            formData.append('temperatura', signosVitales.temperatura);
            // Si backend soporta más enviar resto, sino agregarlos a notas SOAP objetivo.
            const signosDesc = `FC: ${signosVitales.frecuencia_cardiaca} lpm, FR: ${signosVitales.frecuencia_respiratoria}, O2: ${signosVitales.saturacion_oxigeno}%`;

            // JSON fields need to be stringified for FormData if backend doesn't handle array[] for objects well in FormData
            // Laravel handles arrays cleanly in FormData if named correctly: notas_soap[subjetivo]
            formData.append('notas_soap[subjetivo]', soap.subjetivo);
            formData.append('notas_soap[objetivo]', signosDesc + ' | ' + soap.objetivo);
            formData.append('notas_soap[analisis]', soap.analisis);
            formData.append('notas_soap[plan]', soap.plan + (indicacionesGenerales ? `\n\nIndicaciones Geales:\n${indicacionesGenerales}` : '') + (proximaCita ? `\n\nPróxima cita planificada para: ${proximaCita}` : ''));

            let extraList = cie10 ? diagnosticosList : diagnosticosList.slice(1);
            let additionalDiags = extraList.map(d => `${d.cie10.codigo} - ${d.cie10.descripcion} (${d.tipo})`).join(', ');
            if (additionalDiags) formData.append('notas_soap[analisis]', soap.analisis + `\n\nDiagnósticos Secundarios: ${additionalDiags}`);

            // Examen fisico array
            examenFisico.forEach((ex, index) => {
                formData.append(`examen_fisico[${index}][sistema]`, ex.sistema);
                formData.append(`examen_fisico[${index}][hallazgos]`, ex.hallazgos);
            });

            // Archivos
            archivos.forEach((file) => {
                formData.append('archivos[]', file);
            });

            // Receta Médica
            if (medicamentos.length > 0) {
                formData.append('receta_instrucciones', recetaInstrucciones);
                medicamentos.forEach((med, i) => {
                    formData.append(`receta_medicamentos[${i}][nombre_medicamento]`, med.nombre_medicamento);
                    formData.append(`receta_medicamentos[${i}][dosis]`, med.dosis);
                    formData.append(`receta_medicamentos[${i}][frecuencia]`, med.frecuencia);
                    formData.append(`receta_medicamentos[${i}][duracion_dias]`, med.duracion_dias.toString());
                    formData.append(`receta_medicamentos[${i}][cantidad_total]`, med.cantidad_total.toString());
                    if (med.instrucciones_especiales) {
                        formData.append(`receta_medicamentos[${i}][instrucciones_especiales]`, med.instrucciones_especiales);
                    }
                });
            }

            // Exámenes / Laboratorios
            const laboratoriosActivos = ordenesLab.filter(o => o.checked).map(o => o.name);
            laboratoriosActivos.forEach((lab, i) => {
                formData.append(`ordenes_laboratorio[${i}]`, lab);
            });

            await axios.post('/historias-clinicas', formData, {
                headers: { 'Content-Type': 'multipart/form-data' }
            });

            // FASE 4: Integración automática: Crear la próxima cita si se especificó fecha
            if (proximaCita) {
                try {
                    await axios.post('/citas', {
                        paciente_id: selectedPaciente.id,
                        medico_id: user?.id || 1,
                        fecha_cita: `${proximaCita} 08:00:00`, // Default morning
                        duracion_minutos: 30,
                        tipo_cita: 'control',
                        motivo_consulta: `Control médico programado automáticamente desde Historia Clínica anterior. ${indicacionesGenerales ? 'Indicación: ' + indicacionesGenerales : ''}`.substring(0, 200),
                        pagado: false
                    });
                    alert(`Historia Clínica guardada con éxito y se programó la Cita de Control para el ${proximaCita}.`);
                } catch (e) {
                    console.error("Error al agendar próxima cita automática:", e);
                    alert("Historia guardada, pero ocurrió un error al programar la próxima cita automática.");
                }
            } else {
                alert('Historia Clínica guardada con éxito!');
            }

            setIsCreating(false);
            // Recargar historial
            const res = await axios.get(`/historias-clinicas?paciente_id=${selectedPaciente.id}`);
            setHistorial(res.data.data);
            // Limpiar citas pendientes
            setCitasPendientes(prev => prev.filter(c => c.id !== selectedCitaId));

        } catch (error) {
            console.error('Error guardando historia:', error);
            alert('Error al guardar la historia. Verifique los datos.');
        } finally {
            setLoading(false);
        }
    };

    const handleEditHistoria = async (item: HistoriaClinica) => {
        const createdDate = new Date(item.created_at);
        const mdDiff = new Date().getTime() - createdDate.getTime();
        const hoursDiff = mdDiff / (1000 * 60 * 60);

        let nota_aclaratoria = '';
        if (hoursDiff > 24) {
            const nota = window.prompt("⚠️ Esta historia superó las 24 horas y está bloqueada por auditoría médica.\n\nPara modificarla, establezca obligatoriamente una nota aclaratoria justificada:");
            if (!nota) return alert("❌ Edición cancelada. Nota aclaratoria vacía.");
            nota_aclaratoria = nota;
        } else {
            const confirm = window.confirm("¿Desea editar este registro?");
            if (!confirm) return;
        }

        try {
            // Ejemplo de endpoint si existiera un PATCH
            await axios.patch(`/historias-clinicas/${item.id}`, {
                nota_aclaratoria: nota_aclaratoria,
                // enviaría data real
            });
            alert('Registro actualizado correctamente.');
            // Refrescar historial
            if (selectedPaciente) {
                const resHist = await axios.get(`/historias-clinicas?paciente_id=${selectedPaciente.id}`);
                setHistorial(resHist.data.data);
            }
        } catch (error: any) {
            if (error.response?.status === 403) {
                alert(`Error de Seguridad 403: ${error.response.data.error || 'Requiere nota_aclaratoria o tiempo expirado.'}`);
            } else {
                console.error(error);
                alert('No se procesó la actualización.');
            }
        }
    };

    const handleImprimirReceta = (item: HistoriaClinica) => {
        if (!item.recetasMedicas || item.recetasMedicas.length === 0) {
            alert("No hay recetas disponibles para esta consulta.");
            return;
        }

        const receta = item.recetasMedicas[0]; // Tomamos la primera receta por simplicidad

        const printWindow = window.open('', '_blank');
        if (!printWindow) return;

        printWindow.document.write(`
            <html>
            <head>
                <title>Receta Médica - ${selectedPaciente?.nombre} ${selectedPaciente?.apellido}</title>
                <style>
                    body { font-family: 'Arial', sans-serif; padding: 40px; color: #333; }
                    .header { text-align: center; border-bottom: 2px solid #2563eb; padding-bottom: 20px; margin-bottom: 30px; }
                    .clinic-name { font-size: 24px; font-weight: bold; color: #2563eb; margin: 0; }
                    .dr-name { font-size: 18px; font-weight: bold; margin-top: 10px; }
                    .info-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 30px; font-size: 14px; }
                    .rx-title { font-size: 32px; font-weight: bold; color: #000; margin-bottom: 20px; }
                    table { w-full: 100%; width: 100%; border-collapse: collapse; margin-bottom: 30px; }
                    th, td { padding: 10px; border-bottom: 1px solid #ddd; text-align: left; }
                    th { font-weight: bold; color: #555; }
                    .instructions { background: #f9fafb; padding: 15px; border-radius: 8px; border: 1px dashed #ccc; }
                    .footer { margin-top: 80px; text-align: center; }
                    .signature-line { width: 300px; border-top: 1px solid #000; margin: 0 auto; padding-top: 10px; }
                    @media print {
                        body { padding: 0; }
                        button { display: none; }
                    }
                </style>
            </head>
            <body>
                <div class="header">
                    <h1 class="clinic-name">Cibertronia Salud</h1>
                    <div class="dr-name">Dr. ${item.medico?.nombre} ${item.medico?.apellido}</div>
                    <div>${item.medico?.especialidad || 'Médico General'}</div>
                </div>

                <div class="info-grid">
                    <div><b>Paciente:</b> ${selectedPaciente?.nombre} ${selectedPaciente?.apellido}</div>
                    <div><b>Fecha:</b> ${new Date(receta.fecha_emision).toLocaleDateString()}</div>
                    <div><b>Documento:</b> ${selectedPaciente?.numero_documento}</div>
                    <div><b>Edad:</b> ${selectedPaciente?.edad || '-'} años</div>
                </div>

                <div class="rx-title">Rx</div>

                <table>
                    <thead>
                        <tr>
                            <th>Medicamento</th>
                            <th>Dosis / Frecuencia</th>
                            <th>Duración</th>
                            <th>Cantidad</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${receta.medicamentos?.map(med => `
                            <tr>
                                <td><b>${med.nombre_medicamento}</b><br><small>${med.via_administracion}</small></td>
                                <td>${med.dosis} - ${med.frecuencia}</td>
                                <td>${med.duracion_dias} días</td>
                                <td># ${med.cantidad_total}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>

                ${receta.instrucciones_generales ? `
                    <div class="instructions">
                        <b>Instrucciones Generales:</b><br/>
                        ${receta.instrucciones_generales}
                    </div>
                ` : ''}

                <div class="footer">
                    <div class="signature-line">
                        Firma y Sello del Médico
                    </div>
                </div>

                <script>
                    setTimeout(() => {
                        window.print();
                    }, 500);
                </script>
            </body>
            </html>
        `);
        printWindow.document.close();
    };

    return (
        <div className="flex h-[calc(100vh-140px)] gap-6">

            {/* --- PANEL IZQUIERDO: LISTA DE PACIENTES --- */}
            <div className={`w-1/3 flex flex-col rounded-2xl ${isDarkMode ? 'bg-gray-800 border-gray-700' : 'bg-white border-gray-200'} border shadow-lg overflow-hidden`}>
                <div className="p-4 border-b border-gray-200 dark:border-gray-700">
                    <h2 className={`text-lg font-bold mb-3 ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>Pacientes</h2>
                    <input
                        type="text"
                        placeholder="Buscar paciente..."
                        className="w-full p-2 rounded-lg border border-gray-300 dark:border-gray-600 dark:bg-gray-700"
                        value={searchQuery}
                        onChange={(e) => setSearchQuery(e.target.value)}
                    />
                </div>

                <div className="flex-1 overflow-y-auto p-2 space-y-2">
                    {pacientes.map(p => (
                        <div
                            key={p.id}
                            onClick={() => setSelectedPaciente(p)}
                            className={`p-3 rounded-xl cursor-pointer transition-all flex items-center gap-3
                                ${selectedPaciente?.id === p.id
                                    ? 'bg-blue-500 text-white shadow-md'
                                    : isDarkMode ? 'hover:bg-gray-700 text-gray-200' : 'hover:bg-gray-50 text-gray-800'}
                            `}
                        >
                            <div className={`w-10 h-10 rounded-full flex items-center justify-center font-bold
                                ${selectedPaciente?.id === p.id ? 'bg-white text-blue-600' : 'bg-gray-200 text-gray-600'}
                            `}>
                                {p.nombre.charAt(0)}
                            </div>
                            <div>
                                <p className="font-bold">{p.nombre} {p.apellido}</p>
                                <p className={`text-xs ${selectedPaciente?.id === p.id ? 'text-blue-100' : 'text-gray-500'}`}>
                                    DNI: {p.numero_documento}
                                </p>
                            </div>
                        </div>
                    ))}
                </div>
            </div>

            {/* --- PANEL DERECHO: DETALLE / NUEVA CONSULTA --- */}
            <div className={`flex-1 rounded-2xl ${isDarkMode ? 'bg-gray-800 border-gray-700' : 'bg-white border-gray-200'} border shadow-lg overflow-hidden flex flex-col`}>
                {!selectedPaciente ? (
                    <div className="flex-1 flex flex-col items-center justify-center text-gray-400 p-10 text-center">
                        <div className="text-6xl mb-4">👈</div>
                        <p>Seleccione un paciente para ver su historial o iniciar consulta.</p>
                    </div>
                ) : (
                    <>
                        {/* Header del Paciente */}
                        <div className="p-6 border-b border-gray-200 dark:border-gray-700 flex justify-between items-center bg-gray-50 dark:bg-gray-800">
                            <div className="flex items-center gap-4">
                                {selectedPaciente.foto_paciente ? (
                                    <img src={selectedPaciente.foto_paciente} alt="Foto paciente" className="w-16 h-16 rounded-full object-cover border-2 border-blue-500" />
                                ) : (
                                    <div className="w-16 h-16 rounded-full bg-blue-100 flex items-center justify-center text-blue-600 text-2xl font-bold">
                                        {selectedPaciente.nombre.charAt(0)}{selectedPaciente.apellido.charAt(0)}
                                    </div>
                                )}
                                <div>
                                    <h2 className={`text-2xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'} flex items-center gap-2`}>
                                        {selectedPaciente.nombre} {selectedPaciente.apellido}
                                        {selectedPaciente.edad !== undefined && (
                                            <span className="text-sm bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 px-2 py-1 rounded-full">
                                                {selectedPaciente.edad} años
                                            </span>
                                        )}
                                    </h2>
                                    <p className="text-gray-500 text-sm mt-1">
                                        Historia Clínica Digital
                                        {selectedPaciente.grupo_sanguineo && (
                                            <span className="ml-3 font-semibold text-red-600 dark:text-red-400">
                                                🩸 {selectedPaciente.grupo_sanguineo}
                                            </span>
                                        )}
                                    </p>
                                    {selectedPaciente.alergias && selectedPaciente.alergias.trim() !== '' && (
                                        <div className="mt-2 text-xs font-bold text-red-700 bg-red-100 border border-red-300 rounded-md px-2 py-1 inline-block">
                                            ⚠️ Alergias: {selectedPaciente.alergias}
                                        </div>
                                    )}
                                </div>
                            </div>

                            {/* Botón Nueva Consulta (solo si no estamos creando ya) */}
                            {!isCreating && (
                                <div className="flex gap-2">
                                    {citasPendientes.length > 0 ? (
                                        citasPendientes.map(cita => (
                                            <button
                                                key={cita.id}
                                                onClick={() => handleStartConsultation(cita)}
                                                className="px-4 py-2 bg-green-500 hover:bg-green-600 text-white rounded-lg shadow-lg font-bold flex items-center gap-2 animate-pulse"
                                            >
                                                🩺 Atender Cita {cita.fecha_cita.split(' ')[1]?.substring(0, 5)}
                                            </button>
                                        ))
                                    ) : (
                                        <button className="px-4 py-2 bg-gray-300 text-gray-600 rounded-lg cursor-not-allowed" disabled>
                                            No hay citas pendientes
                                        </button>
                                    )}
                                </div>
                            )}
                        </div>

                        {/* CONTENIDO PRINCIPAL */}
                        <div className="flex-1 overflow-y-auto p-6">
                            {isCreating ? (
                                /* --- MODO: NUEVA CONSULTA --- */
                                <div className="space-y-8 animate-fadeIn">
                                    <div className="flex justify-between items-center">
                                        <h3 className="text-xl font-bold text-blue-600">Nueva Consulta Médica</h3>
                                        <button onClick={() => setIsCreating(false)} className="text-gray-500 hover:text-gray-700">Cancelar</button>
                                    </div>

                                    {/* SECCIÓN 1: DATOS DE FILIACIÓN (Resumen) */}
                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
                                        <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-4">👤 Sección 1: Datos de Filiación</h3>
                                        <div className="grid grid-cols-2 lg:grid-cols-4 gap-4 text-sm">
                                            <div>
                                                <p className="text-gray-500 font-bold">Documento</p>
                                                <p>{selectedPaciente.numero_documento}</p>
                                            </div>
                                            <div>
                                                <p className="text-gray-500 font-bold">Fecha de Nacimiento</p>
                                                <p>{selectedPaciente.fecha_nacimiento || 'No registrada'}</p>
                                            </div>
                                            <div>
                                                <p className="text-gray-500 font-bold">Contacto de Emergencia</p>
                                                <p>{selectedPaciente.contacto_emergencia || 'No registrado'}</p>
                                            </div>
                                            <div>
                                                <p className="text-gray-500 font-bold">Telf. Emergencia</p>
                                                <p>{selectedPaciente.telefono_emergencia || 'No registrado'}</p>
                                            </div>
                                        </div>
                                    </div>

                                    {/* SECCIÓN 2: ANAMNESIS */}
                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
                                        <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-4">🗣️ Sección 2: Anamnesis</h3>
                                        <div className="space-y-4">
                                            <div>
                                                <label className="block text-sm font-bold mb-1">Motivo de Consulta</label>
                                                <input
                                                    type="text"
                                                    className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700"
                                                    value={motivoConsulta}
                                                    onChange={e => setMotivoConsulta(e.target.value)}
                                                    placeholder="Ej: Dolor de cabeza intenso..."
                                                />
                                            </div>
                                            <div>
                                                <label className="block text-sm font-bold mb-1">Enfermedad Actual (Subjetivo SOAP)</label>
                                                <textarea
                                                    className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 min-h-[100px]"
                                                    value={soap.subjetivo}
                                                    onChange={e => setSoap(prev => ({ ...prev, subjetivo: e.target.value }))}
                                                    placeholder="Descripción detallada de la enfermedad actual, tiempo de evolución, síntomas acompañantes..."
                                                />
                                            </div>
                                        </div>
                                    </div>

                                    {/* SECCIÓN 3: ANTECEDENTES */}
                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
                                        <div className="flex justify-between items-center mb-4">
                                            <h3 className="text-lg font-bold text-gray-900 dark:text-white">📋 Sección 3: Antecedentes Médico/Personales</h3>
                                        </div>
                                        <div className="space-y-3">
                                            {antecedentes.length > 0 ? antecedentes.map((ant, i) => (
                                                <div key={i} className="flex gap-2 items-start p-3 bg-gray-50 dark:bg-gray-700 rounded border border-gray-200 dark:border-gray-600">
                                                    <span className="bg-blue-100 text-blue-800 font-bold text-xs px-2 py-1 rounded">{ant.tipo}</span>
                                                    <div className="flex-1">
                                                        <p className="font-semibold text-sm text-gray-800 dark:text-white">{ant.descripcion}</p>
                                                        {ant.observaciones && <p className="text-xs text-gray-500 mt-1">{ant.observaciones}</p>}
                                                    </div>
                                                </div>
                                            )) : (
                                                <div className="text-sm text-gray-500 italic">No se encontraron antecedentes patológicos registrados en la ficha del paciente.</div>
                                            )}
                                        </div>
                                    </div>

                                    {/* SECCIÓN 4: SIGNOS VITALES */}
                                    <SignosVitalesForm signos={signosVitales} onChange={setSignosVitales} />

                                    {/* SECCIÓN 5: SOAP (Restante) y EXAMEN FÍSICO */}
                                    <SoapForm
                                        soap={soap}
                                        onChangeSoap={setSoap}
                                        examenFisico={examenFisico}
                                        onChangeExamen={setExamenFisico}
                                    />

                                    {/* DIAGNÓSTICO (Vinculado indirectamente) */}
                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
                                        <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-4">🩺 Sección 6: Diagnóstico (CIE-10)</h3>
                                        <div>
                                            <Cie10Select onSelect={setCie10} selectedId={cie10?.id} />
                                            <div className="mt-4 flex gap-4 items-center">
                                                <label className="flex items-center gap-2 cursor-pointer">
                                                    <input type="radio" name="dg_type" checked={diagnosticoTipo === 'presuntivo'} onChange={() => setDiagnosticoTipo('presuntivo')} />
                                                    <span className="font-bold">Presuntivo</span>
                                                </label>
                                                <label className="flex items-center gap-2 cursor-pointer">
                                                    <input type="radio" name="dg_type" checked={diagnosticoTipo === 'definitivo'} onChange={() => setDiagnosticoTipo('definitivo')} />
                                                    <span className="font-bold">Definitivo</span>
                                                </label>
                                                <button
                                                    onClick={() => {
                                                        if (cie10) {
                                                            setDiagnosticosList(prev => [...prev, { cie10, tipo: diagnosticoTipo }]);
                                                            setCie10(null);
                                                        }
                                                    }}
                                                    className="ml-auto px-4 py-2 bg-blue-100 text-blue-700 font-bold rounded-lg"
                                                >
                                                    + Añadir Diagnóstico
                                                </button>
                                            </div>

                                            {/* LISTA DE DIAGNÓSTICOS AGREGADOS */}
                                            {diagnosticosList.length > 0 && (
                                                <div className="mt-4 space-y-2 border-t pt-4">
                                                    <p className="font-bold text-sm text-gray-500">Diagnósticos Registrados:</p>
                                                    {diagnosticosList.map((d, index) => (
                                                        <div key={index} className="flex justify-between items-center bg-gray-50 dark:bg-gray-700 p-2 rounded border">
                                                            <div>
                                                                <span className="font-bold text-blue-600 mr-2">{d.cie10.codigo}</span>
                                                                <span>{d.cie10.descripcion}</span>
                                                                <span className="ml-2 text-xs uppercase bg-gray-200 dark:bg-gray-600 px-2 py-1 rounded">{d.tipo}</span>
                                                            </div>
                                                            <button className="text-red-500 hover:bg-red-100 p-1 rounded" onClick={() => setDiagnosticosList(prev => prev.filter((_, i) => i !== index))}>🗑️</button>
                                                        </div>
                                                    ))}
                                                </div>
                                            )}
                                        </div>
                                    </div>

                                    {/* 7. Plan y Tratamiento */}
                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
                                        <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-4">💊 Sección 7: Plan y Tratamiento</h3>

                                        <RecetaForm
                                            medicamentos={medicamentos}
                                            onChangeMedicamentos={setMedicamentos}
                                            instrucciones={recetaInstrucciones}
                                            onChangeInstrucciones={setRecetaInstrucciones}
                                        />

                                        <div className="mt-6 grid grid-cols-1 md:grid-cols-2 gap-6 border-t pt-6">
                                            {/* Ordenes de Lab */}
                                            <div>
                                                <label className="block text-sm font-bold mb-2">Órdenes de Laboratorio</label>
                                                <div className="space-y-2 bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                                                    {ordenesLab.map((ol, idx) => (
                                                        <label key={ol.id} className="flex items-center gap-2 cursor-pointer">
                                                            <input type="checkbox" checked={ol.checked} onChange={e => {
                                                                const newOl = [...ordenesLab];
                                                                newOl[idx].checked = e.target.checked;
                                                                setOrdenesLab(newOl);
                                                            }} />
                                                            <span>{ol.name}</span>
                                                        </label>
                                                    ))}
                                                </div>
                                            </div>

                                            {/* Indicaciones Generales & Cita */}
                                            <div className="space-y-4">
                                                <div>
                                                    <label className="block text-sm font-bold mb-2">Indicaciones Generales</label>
                                                    <textarea
                                                        className="w-full p-2 border rounded-lg dark:bg-gray-700 min-h-[80px]"
                                                        placeholder="Descanso, dieta, etc."
                                                        value={indicacionesGenerales}
                                                        onChange={e => setIndicacionesGenerales(e.target.value)}
                                                    />
                                                </div>
                                                <div>
                                                    <label className="block text-sm font-bold mb-2">Programar Próxima Cita</label>
                                                    <input
                                                        type="date"
                                                        className="w-full p-2 border rounded-lg dark:bg-gray-700"
                                                        value={proximaCita}
                                                        onChange={e => setProximaCita(e.target.value)}
                                                    />
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    {/* 8. Archivos */}
                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
                                        <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-4">📂 Sección 8: Archivos Adjuntos</h3>
                                        <FileUploader files={archivos} onFilesChange={setArchivos} />
                                    </div>

                                    {/* Footer Actions */}
                                    <div className="pt-6 border-t border-gray-200 dark:border-gray-700 flex justify-end gap-4">
                                        <button
                                            onClick={() => setIsCreating(false)}
                                            className="px-6 py-3 rounded-xl border border-gray-300 hover:bg-gray-50 dark:border-gray-600 dark:hover:bg-gray-700"
                                        >
                                            Cancelar
                                        </button>
                                        <button
                                            onClick={handleSaveHistoria}
                                            disabled={loading}
                                            className="px-8 py-3 rounded-xl bg-gradient-to-r from-blue-600 to-cyan-500 text-white font-bold shadow-lg hover:shadow-xl transform hover:scale-105 transition-all disabled:opacity-50"
                                        >
                                            {loading ? 'Guardando...' : '💾 Finalizar Consulta'}
                                        </button>
                                    </div>
                                </div>
                            ) : (
                                /* --- MODO: VER HISTORIAL --- */
                                <div className="space-y-6">
                                    {historial.length === 0 ? (
                                        <div className="text-center py-10 text-gray-500 border-2 border-dashed rounded-xl">
                                            No hay registros históricos para este paciente.
                                        </div>
                                    ) : (
                                        historial.map((item, idx) => (
                                            <div key={item.id} className="relative pl-8 pb-8 border-l-2 border-blue-200 dark:border-blue-900 last:pb-0">
                                                {/* Linea de tiempo dot */}
                                                <div className="absolute -left-[9px] top-0 w-4 h-4 rounded-full bg-blue-500 border-4 border-white dark:border-gray-800"></div>

                                                <div className="bg-white dark:bg-gray-800 rounded-xl p-5 shadow-sm border border-gray-100 dark:border-gray-700 hover:shadow-md transition-shadow">
                                                    <div className="flex justify-between items-start mb-3">
                                                        <div>
                                                            <span className="text-xs font-bold uppercase tracking-wider text-blue-600 bg-blue-50 px-2 py-1 rounded-md">
                                                                {new Date(item.fecha_consulta).toLocaleDateString()}
                                                            </span>
                                                            <span className={`ml-2 text-xs font-bold px-2 py-1 rounded-md inline-block ${idx === historial.length - 1 ? 'bg-purple-100 text-purple-700' : 'bg-gray-100 text-gray-700'}`}>
                                                                {idx === historial.length - 1 ? '🌟 Primera Consulta' : '🔄 Seguimiento'}
                                                            </span>
                                                            <h4 className="text-lg font-bold mt-2 text-gray-900 dark:text-white">
                                                                {item.motivo_consulta}
                                                            </h4>
                                                        </div>
                                                        <div className="text-right">
                                                            <div className="text-sm font-medium text-gray-900 dark:text-gray-300">👨‍⚕️ Dr. {item.medico?.apellido}</div>
                                                            <div className="text-xs text-gray-500">{item.medico?.especialidad}</div>
                                                            <div className="mt-1 text-xs text-gray-400 italic">Creado el {new Date(item.created_at).toLocaleString()}</div>
                                                            <div className="mt-2 flex gap-2 justify-end">
                                                                <button
                                                                    className="text-xs font-bold px-2 py-1 bg-yellow-100 hover:bg-yellow-200 text-yellow-700 rounded transition-colors"
                                                                    onClick={() => handleEditHistoria(item)}
                                                                >
                                                                    ✏️ Editar
                                                                </button>
                                                                {item.recetasMedicas && item.recetasMedicas.length > 0 && (
                                                                    <button
                                                                        className="text-xs font-bold px-2 py-1 bg-green-100 hover:bg-green-200 text-green-700 rounded transition-colors"
                                                                        onClick={() => handleImprimirReceta(item)}
                                                                    >
                                                                        💊 Imprimir Receta
                                                                    </button>
                                                                )}
                                                                <button
                                                                    className="text-xs font-bold px-2 py-1 bg-gray-100 hover:bg-gray-200 text-gray-700 rounded transition-colors"
                                                                    onClick={() => alert('WIP: Generando PDF...')}
                                                                >
                                                                    📄 Exportar PDF
                                                                </button>
                                                                <button
                                                                    className="text-xs font-bold px-2 py-1 bg-gray-100 hover:bg-gray-200 text-gray-700 rounded transition-colors"
                                                                    onClick={() => {
                                                                        const confirm = window.confirm("¿Desea imprimir esta consulta?");
                                                                        if (confirm) window.print();
                                                                    }}
                                                                >
                                                                    🖨️ Imprimir
                                                                </button>
                                                                <button
                                                                    className="text-xs font-bold px-2 py-1 bg-blue-100 hover:bg-blue-200 text-blue-700 rounded transition-colors"
                                                                    onClick={() => alert('Vista detallada completa (WIP - se abriría el expediente en un modal o nueva ruta)')}
                                                                >
                                                                    🔍 Ver Detalle
                                                                </button>
                                                            </div>

                                                            {/* Miniatura de Exámenes Solicitados */}
                                                            {item.examenesMedicos && item.examenesMedicos.length > 0 && (
                                                                <div className="mt-3 bg-blue-50/50 p-2 rounded-lg border border-blue-100 text-left">
                                                                    <div className="text-xs font-bold text-blue-800 mb-1">🔬 Exámenes Solicitados:</div>
                                                                    <div className="flex flex-wrap gap-1">
                                                                        {item.examenesMedicos.map((ex: any) => (
                                                                            <span key={ex.id} className="inline-flex items-center px-2 py-0.5 rounded text-[10px] font-medium bg-blue-100 text-blue-800">
                                                                                {ex.descripcion}
                                                                                {ex.estado === 'completado' ? (
                                                                                    <button
                                                                                        onClick={() => window.open(ex.archivo_resultado ? `/storage/examenes/${ex.archivo_resultado}` : '#', '_blank')}
                                                                                        className="ml-1 text-green-600 hover:text-green-800 underline"
                                                                                        title="Ver Resultados"
                                                                                    >
                                                                                        (Ver)
                                                                                    </button>
                                                                                ) : (
                                                                                    <span className="ml-1 text-gray-400">({ex.estado})</span>
                                                                                )}
                                                                            </span>
                                                                        ))}
                                                                    </div>
                                                                </div>
                                                            )}

                                                        </div>
                                                    </div>

                                                    {/* Diagnóstico */}
                                                    {item.cie10 && (
                                                        <div className="mb-4 p-3 bg-red-50 dark:bg-red-900/10 border-l-4 border-red-500 rounded-r-lg">
                                                            <span className="font-bold text-red-700 dark:text-red-400">{item.cie10.codigo}</span>: {item.cie10.descripcion}
                                                            <span className="text-xs ml-2 uppercase text-gray-500">({item.tipo_diagnostico})</span>
                                                        </div>
                                                    )}

                                                    {/* SOAP Mini View */}
                                                    {item.notas_soap && (
                                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm mt-4 bg-gray-50 dark:bg-gray-700/30 p-4 rounded-lg">
                                                            <div><span className="font-bold text-gray-700 dark:text-gray-300">Subjetivo:</span> <p>{item.notas_soap.subjetivo || '-'}</p></div>
                                                            <div><span className="font-bold text-gray-700 dark:text-gray-300">Objetivo:</span> <p>{item.notas_soap.objetivo || '-'}</p></div>
                                                            <div><span className="font-bold text-gray-700 dark:text-gray-300">Análisis:</span> <p>{item.notas_soap.analisis || '-'}</p></div>
                                                            <div><span className="font-bold text-gray-700 dark:text-gray-300">Plan:</span> <p>{item.notas_soap.plan || '-'}</p></div>
                                                        </div>
                                                    )}

                                                    {/* Archivos */}
                                                    {item.archivos && item.archivos.length > 0 && (
                                                        <div className="mt-4 flex gap-2 overflow-x-auto pb-2">
                                                            {item.archivos.map(arch => (
                                                                <a
                                                                    key={arch.id}
                                                                    href={`/storage/${arch.ruta_archivo}`}
                                                                    target="_blank"
                                                                    className="flex items-center gap-2 px-3 py-2 bg-gray-100 dark:bg-gray-700 rounded-lg text-xs hover:bg-blue-100 transition-colors"
                                                                >
                                                                    📎 {arch.nombre_original}
                                                                </a>
                                                            ))}
                                                        </div>
                                                    )}
                                                </div>
                                            </div>
                                        ))
                                    )}
                                </div>
                            )}
                        </div>
                    </>
                )}
            </div>
        </div>
    );
}
