import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { useUser } from '../../context/UserContext';
import { useTheme } from '../../context/ThemeContext';
import axios from 'axios';

export default function Login() {
    const [email, setEmail] = useState('');
    const [password, setPassword] = useState('');
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState('');
    const { login } = useUser();
    const { theme, toggleTheme } = useTheme();
    const navigate = useNavigate();
    const isDarkMode = theme === 'dark';

    const handleLogin = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');
        setLoading(true);

        try {
            const response = await axios.post('/login', {
                email,
                password
            });

            if (response.data.success) {
                // Guardar token y usuario
                localStorage.setItem('token', response.data.token);
                localStorage.setItem('user', JSON.stringify(response.data.user));
                
                // Configurar axios para futuras peticiones
                axios.defaults.headers.common['Authorization'] = `Bearer ${response.data.token}`;

                // Actualizar contexto
                login(response.data.user);

                // Redirigir al dashboard
                navigate('/dashboard');
            }
        } catch (err: any) {
            setError(err.response?.data?.message || 'Error al iniciar sesión');
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className={`min-h-screen flex items-center justify-center p-4 transition-colors duration-300 ${isDarkMode 
            ? 'bg-gradient-to-br from-gray-900 via-slate-900 to-zinc-900' 
            : 'bg-gradient-to-br from-blue-600 via-cyan-500 to-teal-400'}`}>
            <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjAiIGhlaWdodD0iNjAiIHZpZXdCb3g9IjAgMCA2MCA2MCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48ZyBmaWxsPSJub25lIiBmaWxsLXJ1bGU9ImV2ZW5vZGQiPjxnIGZpbGw9IiNmZmYiIGZpbGwtb3BhY2l0eT0iMC4wNSI+PHBhdGggZD0iTTM2IDEzNGgtMTBWMTVoMTB6TTI1IDE1SDE1djEwaDEwem0tMTAgMEg1djEwaDEwem0wLTEwSDB2MTBoNXYtNXoiLz48L2c+PC9nPjwvc3ZnPg==')] opacity-20"></div>

            {/* Theme Toggle Button */}
            <button
                onClick={toggleTheme}
                className={`fixed top-6 right-6 p-3 rounded-2xl transition-all hover:scale-110 z-50 shadow-lg backdrop-blur-lg ${isDarkMode 
                    ? 'bg-gray-800/80 hover:bg-gray-700 text-white border border-gray-600' 
                    : 'bg-white/80 hover:bg-white text-gray-800 border border-gray-200'}`}
                title={isDarkMode ? 'Cambiar a modo claro' : 'Cambiar a modo oscuro'}
            >
                {isDarkMode ? '🌙' : '☀️'}
            </button>

            <div className="relative w-full max-w-md">
                {/* Logo Card */}
                <div className={`backdrop-blur-xl rounded-3xl shadow-2xl p-8 mb-6 text-center transition-colors duration-300 ${isDarkMode 
                    ? 'bg-gray-800/95 border border-gray-700' 
                    : 'bg-white/95'}`}>
                    <div className={`w-20 h-20 rounded-2xl mx-auto flex items-center justify-center text-4xl mb-4 shadow-lg transform hover:scale-110 transition-transform ${isDarkMode 
                        ? 'bg-gradient-to-br from-purple-600 to-pink-600' 
                        : 'bg-gradient-to-br from-blue-600 to-cyan-500'}`}>
                        🏥
                    </div>
                    <h1 className={`text-3xl font-bold bg-clip-text text-transparent mb-2 ${isDarkMode 
                        ? 'bg-gradient-to-r from-purple-400 to-pink-400' 
                        : 'bg-gradient-to-r from-blue-600 to-cyan-500'}`}>
                        Cibertronia Salud
                    </h1>
                    <p className={`transition-colors duration-300 ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                        Sistema de Gestión Médica
                    </p>
                </div>

                {/* Login Form */}
                <div className={`backdrop-blur-xl rounded-3xl shadow-2xl p-8 transition-colors duration-300 ${isDarkMode 
                    ? 'bg-gray-800/95 border border-gray-700' 
                    : 'bg-white/95'}`}>
                    <h2 className={`text-2xl font-bold mb-6 transition-colors duration-300 ${isDarkMode ? 'text-white' : 'text-gray-800'}`}>
                        Iniciar Sesión
                    </h2>

                    {error && (
                        <div className="mb-4 p-4 bg-red-50 border-l-4 border-red-500 rounded-lg">
                            <p className="text-red-700 text-sm font-semibold">❌ {error}</p>
                        </div>
                    )}

                    <form onSubmit={handleLogin} className="space-y-6">
                        {/* Email */}
                        <div>
                            <label className={`block text-sm font-semibold mb-2 transition-colors duration-300 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                📧 Correo Electrónico
                            </label>
                            <input
                                type="email"
                                value={email}
                                onChange={(e) => setEmail(e.target.value)}
                                required
                                className={`w-full px-4 py-3 rounded-xl border-2 transition-all outline-none ${isDarkMode 
                                    ? 'bg-gray-700 border-gray-600 text-white placeholder-gray-500 focus:border-purple-500 focus:ring-4 focus:ring-purple-500/20' 
                                    : 'bg-white border-gray-200 text-gray-900 focus:border-blue-500 focus:ring-4 focus:ring-blue-100'}`}
                                placeholder="usuario@cibertroniasalud.com"
                            />
                        </div>

                        {/* Password */}
                        <div>
                            <label className={`block text-sm font-semibold mb-2 transition-colors duration-300 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                🔒 Contraseña
                            </label>
                            <input
                                type="password"
                                value={password}
                                onChange={(e) => setPassword(e.target.value)}
                                required
                                className={`w-full px-4 py-3 rounded-xl border-2 transition-all outline-none ${isDarkMode 
                                    ? 'bg-gray-700 border-gray-600 text-white placeholder-gray-500 focus:border-purple-500 focus:ring-4 focus:ring-purple-500/20' 
                                    : 'bg-white border-gray-200 text-gray-900 focus:border-blue-500 focus:ring-4 focus:ring-blue-100'}`}
                                placeholder="••••••••"
                            />
                        </div>

                        {/* Remember Me */}
                        <div className="flex items-center justify-between">
                            <label className="flex items-center gap-2 cursor-pointer">
                                <input
                                    type="checkbox"
                                    className={`w-4 h-4 rounded transition-colors ${isDarkMode 
                                        ? 'border-gray-600 bg-gray-700 text-purple-600 focus:ring-purple-500' 
                                        : 'border-gray-300 text-blue-600 focus:ring-blue-500'}`}
                                />
                                <span className={`text-sm transition-colors ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>Recordarme</span>
                            </label>
                            <a href="#" className={`text-sm font-semibold transition-colors ${isDarkMode 
                                ? 'text-purple-400 hover:text-purple-300' 
                                : 'text-blue-600 hover:text-blue-700'}`}>
                                ¿Olvidaste tu contraseña?
                            </a>
                        </div>

                        {/* Submit Button */}
                        <button
                            type="submit"
                            disabled={loading}
                            className={`w-full py-4 rounded-xl font-bold text-lg shadow-lg hover:shadow-xl transform hover:scale-105 transition-all disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none ${isDarkMode 
                                ? 'bg-gradient-to-r from-purple-600 to-pink-600 text-white' 
                                : 'bg-gradient-to-r from-blue-600 to-cyan-500 text-white'}`}
                        >
                            {loading ? (
                                <span className="flex items-center justify-center gap-2">
                                    <svg className="animate-spin h-5 w-5" viewBox="0 0 24 24">
                                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
                                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                                    </svg>
                                    Iniciando sesión...
                                </span>
                            ) : (
                                'Iniciar Sesión'
                            )}
                        </button>
                    </form>

                    {/* Demo Credentials */}
                    <div className={`mt-6 p-4 rounded-xl border transition-colors duration-300 ${isDarkMode 
                        ? 'bg-purple-900/30 border-purple-700/50' 
                        : 'bg-blue-50 border-blue-200'}`}>
                        <p className={`text-xs font-semibold mb-2 transition-colors ${isDarkMode ? 'text-purple-300' : 'text-blue-800'}`}>
                            👤 Credenciales de prueba:
                        </p>
                        <div className={`text-xs space-y-1 transition-colors ${isDarkMode ? 'text-purple-200' : 'text-blue-700'}`}>
                            <p><strong>Email:</strong> admin@cibertroniasalud.com</p>
                            <p><strong>Contraseña:</strong> Admin123!</p>
                        </div>
                    </div>
                </div>

                {/* Footer */}
                <p className={`text-center text-sm mt-6 transition-colors ${isDarkMode ? 'text-gray-400' : 'text-white/90'}`}>
                    © 2024 Cibertronia Salud. Todos los derechos reservados.
                </p>
            </div>
        </div>
    );
}
