import React, { useState, useEffect } from 'react';
import { useTheme } from '../../context/ThemeContext';
import { useNavigate } from 'react-router-dom';
import { Users, Plus, Search, Edit, Trash2, Eye, FileText, ClipboardList } from 'lucide-react';

interface Paciente {
    id: number;
    numero_documento: string; // Antes ci
    nombre: string;
    apellido: string;
    telefono: string; // Antes celular
    sexo: string;
    direccion: string | null;
    fecha_nacimiento: string | null;
    email: string | null;
    notas: string | null;
    estado: string;
    created_at: string;
    numero_historia?: string;
    // Campos opcionales para compatibilidad
    edad?: number;
}

export default function Pacientes() {
    const { theme } = useTheme();
    const isDarkMode = theme === 'dark';
    const navigate = useNavigate();
    const [pacientes, setPacientes] = useState<Paciente[]>([]);
    const [loading, setLoading] = useState(true);
    const [searchTerm, setSearchTerm] = useState('');
    const [showModal, setShowModal] = useState(false);
    const [selectedPaciente, setSelectedPaciente] = useState<Paciente | null>(null);
    const [formData, setFormData] = useState({
        numero_documento: '',
        nombre: '',
        apellido: '',
        telefono: '',
        sexo: 'M',
        direccion: '',
        fecha_nacimiento: '',
        email: '',
        notas: '',
        estado: 'activo'
    });

    useEffect(() => {
        fetchPacientes();
    }, []);

    const fetchPacientes = async () => {
        try {
            const token = localStorage.getItem('token');
            if (!token) {
                window.location.href = '/login';
                return;
            }
            const response = await fetch('/api/pacientes', {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Accept': 'application/json',
                },
            });
            if (response.status === 401) {
                localStorage.clear();
                window.location.href = '/login';
                return;
            }
            const data = await response.json();
            if (data.success) {
                setPacientes(data.data);
            }
        } catch (error) {
            console.error('Error fetching pacientes:', error);
        } finally {
            setLoading(false);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        try {
            const token = localStorage.getItem('token');
            if (!token) {
                alert('No estás autenticado.');
                return;
            }

            const url = selectedPaciente
                ? `/api/pacientes/${selectedPaciente.id}`
                : '/api/pacientes';
            const method = selectedPaciente ? 'PUT' : 'POST';

            // Mapear formData a lo que espera el backend legacy (si es necesario)
            // El modelo Paciente espera: numero_documento, telefono, sexo (M/F), etc.
            const payload = {
                ...formData,
                // Asegurar que sexo sea M o F si el select dice Masculino/Femenino
                sexo: formData.sexo === 'Masculino' ? 'M' : (formData.sexo === 'Femenino' ? 'F' : formData.sexo)
            };

            const response = await fetch(url, {
                method,
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                body: JSON.stringify(payload),
            });

            const data = await response.json();

            if (data.success) {
                fetchPacientes();
                setShowModal(false);
                resetForm();
                alert('Paciente guardado exitosamente');
            } else {
                alert(data.message || 'Error al guardar paciente');
            }
        } catch (error: any) {
            console.error('Error saving paciente:', error);
            alert('Error al guardar paciente');
        }
    };

    const handleDelete = async (id: number) => {
        if (!confirm('¿Estás seguro de eliminar este paciente?')) return;

        try {
            const token = localStorage.getItem('token');
            const response = await fetch(`/api/pacientes/${id}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Accept': 'application/json',
                },
            });

            const data = await response.json();
            if (data.success) {
                fetchPacientes();
            }
        } catch (error) {
            console.error('Error deleting paciente:', error);
        }
    };

    const openEditModal = (paciente: Paciente) => {
        setSelectedPaciente(paciente);
        setFormData({
            numero_documento: paciente.numero_documento,
            nombre: paciente.nombre,
            apellido: paciente.apellido,
            telefono: paciente.telefono,
            sexo: paciente.sexo,
            direccion: paciente.direccion || '',
            fecha_nacimiento: paciente.fecha_nacimiento || '',
            email: paciente.email || '',
            notas: paciente.notas || '',
            estado: paciente.estado
        });
        setShowModal(true);
    };

    const openCreateModal = () => {
        setSelectedPaciente(null);
        resetForm();
        setShowModal(true);
    };

    const resetForm = () => {
        setFormData({
            numero_documento: '',
            nombre: '',
            apellido: '',
            telefono: '',
            sexo: 'M',
            direccion: '',
            fecha_nacimiento: '',
            email: '',
            notas: '',
            estado: 'activo'
        });
    };

    const filteredPacientes = pacientes.filter(p => {
        const nombreCompleto = `${p.nombre} ${p.apellido}`.toLowerCase();
        const doc = p.numero_documento || '';
        const tel = p.telefono || '';
        const term = searchTerm.toLowerCase();

        return nombreCompleto.includes(term) || doc.includes(term) || tel.includes(term);
    });

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                <div>
                    <h1 className={`text-3xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'} flex items-center`}>
                        <Users className="inline-block w-8 h-8 mr-2" />
                        Pacientes
                    </h1>
                    <p className={`${isDarkMode ? 'text-gray-400' : 'text-gray-600'} mt-1`}>
                        Gestión de pacientes del consultorio
                    </p>
                </div>
            </div>

            {/* Filters */}
            <div className={`rounded-2xl p-6 ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg`}>
                <div className="flex flex-col md:flex-row gap-4">
                    <div className="flex-1 relative">
                        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
                        <input
                            type="text"
                            placeholder="Buscar por nombre, CI o celular..."
                            value={searchTerm}
                            onChange={(e) => setSearchTerm(e.target.value)}
                            className={`w-full pl-10 pr-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 focus:ring-4 focus:ring-blue-100 transition-all outline-none`}
                        />
                    </div>
                    <button
                        onClick={openCreateModal}
                        className="w-full md:w-auto px-6 py-3 bg-gradient-to-r from-blue-500 to-cyan-500 text-white rounded-xl font-semibold shadow-lg hover:shadow-xl transform hover:scale-105 transition-all flex items-center justify-center gap-2"
                    >
                        <Plus className="w-5 h-5" />
                        Nuevo Paciente
                    </button>
                </div>
            </div>

            {/* Mobile Cards View */}
            <div className="md:hidden space-y-4">
                {loading ? (
                    <div className="text-center py-8">
                        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500 mx-auto"></div>
                    </div>
                ) : filteredPacientes.length === 0 ? (
                    <div className={`text-center py-8 ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                        No se encontraron pacientes
                    </div>
                ) : (
                    filteredPacientes.map((paciente) => (
                        <div key={paciente.id} className={`p-4 rounded-xl ${isDarkMode ? 'bg-gray-800 border border-gray-700' : 'bg-white border border-gray-100'} shadow-sm`}>
                            <div className="flex justify-between items-start mb-3">
                                <div className="flex-1 pr-2">
                                    <h3 className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-gray-900'} break-words`}>
                                        {paciente.apellido}, {paciente.nombre}
                                    </h3>
                                    <p className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>
                                        CI: {paciente.numero_documento}
                                    </p>
                                </div>
                                <span className={`px-2 py-1 rounded-lg text-xs font-semibold border ${paciente.estado === 'activo'
                                    ? 'bg-green-100 text-green-800 border-green-300'
                                    : 'bg-gray-100 text-gray-800 border-gray-300'
                                    }`}>
                                    {paciente.estado}
                                </span>
                            </div>

                            <div className="space-y-2 mb-4">
                                <p className={`text-sm ${isDarkMode ? 'text-gray-300' : 'text-gray-600'}`}>
                                    <span className="font-semibold">Teléfono:</span> {paciente.telefono}
                                </p>
                                <p className={`text-sm ${isDarkMode ? 'text-gray-300' : 'text-gray-600'}`}>
                                    <span className="font-semibold">Nacimiento:</span> {paciente.fecha_nacimiento}
                                </p>
                            </div>

                            <div className="flex gap-2 border-t pt-3 dark:border-gray-700">
                                <button
                                    onClick={() => openEditModal(paciente)}
                                    className="p-2 rounded-lg bg-blue-100 hover:bg-blue-200 text-blue-600 transition-colors"
                                >
                                    <Edit className="w-4 h-4" />
                                </button>
                                <button
                                    onClick={() => handleDelete(paciente.id)}
                                    className="p-2 rounded-lg bg-red-100 hover:bg-red-200 text-red-600 transition-colors"
                                    title="Eliminar"
                                >
                                    <Trash2 className="w-4 h-4" />
                                </button>
                                <button
                                    onClick={() => navigate('/historias-clinicas', { state: { pacienteId: paciente.id } })}
                                    className="p-2 ml-auto rounded-lg bg-emerald-100 hover:bg-emerald-200 text-emerald-700 transition-colors flex items-center gap-1 text-xs font-bold"
                                    title="Ver Historia Clínica"
                                >
                                    <ClipboardList className="w-4 h-4" />
                                    Historia
                                </button>
                            </div>
                        </div>
                    ))
                )}
            </div>

            {/* Desktop Table */}
            <div className={`hidden md:block rounded-2xl ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg overflow-hidden`}>
                <div className="overflow-x-auto">
                    <table className="w-full">
                        <thead className={`${isDarkMode ? 'bg-gray-700/50' : 'bg-gray-50'}`}>
                            <tr>
                                <th className={`px-6 py-4 text-left text-sm font-bold ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>C.I.</th>
                                <th className={`px-6 py-4 text-left text-sm font-bold ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Paciente</th>
                                <th className={`px-6 py-4 text-left text-sm font-bold ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Teléfono</th>
                                <th className={`px-6 py-4 text-left text-sm font-bold ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Nacimiento</th>
                                <th className={`px-6 py-4 text-left text-sm font-bold ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Sexo</th>
                                <th className={`px-6 py-4 text-left text-sm font-bold ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Estado</th>
                                <th className={`px-6 py-4 text-left text-sm font-bold ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Acciones</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                            {loading ? (
                                <tr>
                                    <td colSpan={7} className="px-6 py-8 text-center">
                                        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500 mx-auto"></div>
                                    </td>
                                </tr>
                            ) : filteredPacientes.length === 0 ? (
                                <tr>
                                    <td colSpan={7} className={`px-6 py-8 text-center ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                                        No se encontraron pacientes
                                    </td>
                                </tr>
                            ) : (
                                filteredPacientes.map((paciente) => (
                                    <tr key={paciente.id} className={`${isDarkMode ? 'hover:bg-gray-700/30' : 'hover:bg-gray-50'} transition-colors`}>
                                        <td className={`px-6 py-4 ${isDarkMode ? 'text-gray-300' : 'text-gray-900'} font-mono`}>
                                            {paciente.numero_documento}
                                        </td>
                                        <td className="px-6 py-4">
                                            <div>
                                                <p className={`font-semibold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                                                    {paciente.apellido}, {paciente.nombre}
                                                </p>
                                                <p className={`text-xs ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>
                                                    HC: {paciente.numero_historia || 'N/A'}
                                                </p>
                                            </div>
                                        </td>
                                        <td className={`px-6 py-4 ${isDarkMode ? 'text-gray-300' : 'text-gray-900'}`}>
                                            {paciente.telefono}
                                        </td>
                                        <td className={`px-6 py-4 ${isDarkMode ? 'text-gray-300' : 'text-gray-900'}`}>
                                            {paciente.fecha_nacimiento}
                                        </td>
                                        <td className={`px-6 py-4 ${isDarkMode ? 'text-gray-300' : 'text-gray-900'}`}>
                                            {paciente.sexo}
                                        </td>
                                        <td className="px-6 py-4">
                                            <span className={`px-3 py-1 rounded-lg text-xs font-semibold border ${paciente.estado === 'activo'
                                                ? 'bg-green-100 text-green-800 border-green-300'
                                                : 'bg-gray-100 text-gray-800 border-gray-300'
                                                }`}>
                                                {paciente.estado}
                                            </span>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className="flex gap-2">
                                                <button
                                                    onClick={() => navigate('/historias-clinicas', { state: { pacienteId: paciente.id } })}
                                                    className="p-2 rounded-lg bg-emerald-100 hover:bg-emerald-200 text-emerald-700 transition-colors"
                                                    title="Ver Historia Clínica"
                                                >
                                                    <ClipboardList className="w-4 h-4" />
                                                </button>
                                                <button
                                                    onClick={() => openEditModal(paciente)}
                                                    className="p-2 rounded-lg bg-blue-100 hover:bg-blue-200 text-blue-600 transition-colors"
                                                    title="Editar Paciente"
                                                >
                                                    <Edit className="w-4 h-4" />
                                                </button>
                                                <button
                                                    onClick={() => handleDelete(paciente.id)}
                                                    className="p-2 rounded-lg bg-red-100 hover:bg-red-200 text-red-600 transition-colors"
                                                    title="Eliminar Paciente"
                                                >
                                                    <Trash2 className="w-4 h-4" />
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                ))
                            )}
                        </tbody>
                    </table>
                </div>
            </div>

            {/* Modal Form */}
            {showModal && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
                    <div className={`rounded-2xl ${isDarkMode ? 'bg-gray-800 border border-gray-700' : 'bg-white'} shadow-2xl w-full max-w-4xl max-h-[90vh] overflow-y-auto`}>
                        <div className={`p-6 border-b ${isDarkMode ? 'border-gray-700' : 'border-gray-200'}`}>
                            <h2 className={`text-2xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                                {selectedPaciente ? 'Editar Paciente' : 'Nuevo Paciente'}
                            </h2>
                        </div>

                        <form onSubmit={handleSubmit} className="p-6 space-y-6">
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label className={`block text-sm font-semibold mb-2 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        C.I. / Documento *
                                    </label>
                                    <input
                                        type="text"
                                        value={formData.numero_documento}
                                        onChange={(e) => setFormData({ ...formData, numero_documento: e.target.value })}
                                        className={`w-full px-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 outline-none`}
                                        required
                                    />
                                </div>

                                <div>
                                    <label className={`block text-sm font-semibold mb-2 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Teléfono *
                                    </label>
                                    <input
                                        type="text"
                                        value={formData.telefono}
                                        onChange={(e) => setFormData({ ...formData, telefono: e.target.value })}
                                        className={`w-full px-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 outline-none`}
                                        required
                                    />
                                </div>

                                <div>
                                    <label className={`block text-sm font-semibold mb-2 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Nombre *
                                    </label>
                                    <input
                                        type="text"
                                        value={formData.nombre}
                                        onChange={(e) => setFormData({ ...formData, nombre: e.target.value })}
                                        className={`w-full px-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 outline-none`}
                                        required
                                    />
                                </div>

                                <div>
                                    <label className={`block text-sm font-semibold mb-2 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Apellido *
                                    </label>
                                    <input
                                        type="text"
                                        value={formData.apellido}
                                        onChange={(e) => setFormData({ ...formData, apellido: e.target.value })}
                                        className={`w-full px-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 outline-none`}
                                        required
                                    />
                                </div>

                                <div>
                                    <label className={`block text-sm font-semibold mb-2 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Sexo *
                                    </label>
                                    <select
                                        value={formData.sexo}
                                        onChange={(e) => setFormData({ ...formData, sexo: e.target.value })}
                                        className={`w-full px-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 outline-none`}
                                        required
                                    >
                                        <option value="M">Masculino (M)</option>
                                        <option value="F">Femenino (F)</option>
                                    </select>
                                </div>

                                <div>
                                    <label className={`block text-sm font-semibold mb-2 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Fecha de Nacimiento
                                    </label>
                                    <input
                                        type="date"
                                        value={formData.fecha_nacimiento}
                                        onChange={(e) => setFormData({ ...formData, fecha_nacimiento: e.target.value })}
                                        className={`w-full px-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 outline-none`}
                                    />
                                </div>

                                <div className="md:col-span-2">
                                    <label className={`block text-sm font-semibold mb-2 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Dirección
                                    </label>
                                    <input
                                        type="text"
                                        value={formData.direccion}
                                        onChange={(e) => setFormData({ ...formData, direccion: e.target.value })}
                                        className={`w-full px-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 outline-none`}
                                    />
                                </div>

                                <div className="md:col-span-2">
                                    <label className={`block text-sm font-semibold mb-2 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Email
                                    </label>
                                    <input
                                        type="email"
                                        value={formData.email}
                                        onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                                        className={`w-full px-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 outline-none`}
                                    />
                                </div>

                                <div className="md:col-span-2">
                                    <label className={`block text-sm font-semibold mb-2 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Notas
                                    </label>
                                    <textarea
                                        value={formData.notas}
                                        onChange={(e) => setFormData({ ...formData, notas: e.target.value })}
                                        rows={3}
                                        className={`w-full px-4 py-3 rounded-xl border-2 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} focus:border-blue-500 outline-none`}
                                    />
                                </div>
                            </div>

                            <div className="flex justify-end gap-4 pt-4 border-t border-gray-200 dark:border-gray-700">
                                <button
                                    type="button"
                                    onClick={() => setShowModal(false)}
                                    className={`px-6 py-3 rounded-xl font-semibold ${isDarkMode ? 'text-gray-300 hover:text-white' : 'text-gray-600 hover:text-gray-900'}`}
                                >
                                    Cancelar
                                </button>
                                <button
                                    type="submit"
                                    className="px-6 py-3 bg-gradient-to-r from-blue-500 to-cyan-500 text-white rounded-xl font-semibold shadow-lg hover:shadow-xl transition-all"
                                >
                                    {selectedPaciente ? 'Actualizar' : 'Guardar'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
