import React, { useState, useEffect } from 'react';
import { useTheme } from '../../context/ThemeContext';
import { Search, Plus, FileText, Printer, X, User, Calendar, AlertCircle, Pill, Trash2 } from 'lucide-react';
import axios from 'axios';

interface Paciente {
    id: number;
    nombre: string;
    apellido: string;
    numero_documento: string;
}

interface Medico {
    id: number;
    nombre: string;
    apellido: string;
    rol: string;
}

interface Medicamento {
    id: number;
    nombre_generico: string;
    nombre_comercial: string;
    concentracion: string;
    categoria: string;
    precio_bs: number;
    via_administracion: string;
    presentacion: string;
    alternativas: string[];
    dosis_adulto: string;
}

interface MedicamentoReceta {
    nombre_medicamento: string;
    dosis: string;
    via_administracion: string;
    frecuencia: string;
    duracion_dias: number;
    cantidad_total: number;
    instrucciones_especiales?: string;
}

interface Receta {
    id: number;
    paciente: Paciente;
    medico: Medico;
    fecha_emision: string;
    vigencia_dias: number;
    estado: string;
    instrucciones_generales: string;
    medicamentos: MedicamentoReceta[];
}

const categorias = [
    { value: '', label: 'Todas las categorías' },
    { value: 'cardiovascular', label: '💊 Cardiovasculares' },
    { value: 'analgesico', label: '💊 Analgésicos/Antiinflamatorios' },
    { value: 'antibiotico', label: '💊 Antibióticos' },
    { value: 'respiratorio', label: '💊 Respiratorios' },
    { value: 'diabetes', label: '💊 Diabetes' },
    { value: 'gastrointestinal', label: '💊 Gastrointestinales' },
    { value: 'psiquiatrico', label: '💊 Psiquiátricos/Neurológicos' },
    { value: 'vitaminas', label: '💊 Vitaminas/Suplementos' },
];

const viasAdministracion = [
    { value: 'oral', label: 'Oral' },
    { value: 'iv', label: 'Intravenosa (IV)' },
    { value: 'im', label: 'Intramuscular (IM)' },
    { value: 'sc', label: 'Subcutánea (SC)' },
    { value: 'topical', label: 'Tópica' },
    { value: 'nasal', label: 'Nasal' },
    { value: 'inhalatoria', label: 'Inhalatoria' },
    { value: 'sublingual', label: 'Sublingual' },
    { value: 'rectal', label: 'Rectal' },
];

const frecuencias = [
    'Cada 4 horas',
    'Cada 6 horas',
    'Cada 8 horas',
    'Cada 12 horas',
    'Cada 24 horas',
    'Cada 8-12 horas',
    'Según dolor',
    'Al acostarse',
    'En ayunas',
    'Con las comidas',
];

export default function RecetasMedicas() {
    const { theme } = useTheme();
    const isDarkMode = theme === 'dark';

    const [recetas, setRecetas] = useState<Receta[]>([]);
    const [pacientes, setPacientes] = useState<Paciente[]>([]);
    const [medicos, setMedicos] = useState<Medico[]>([]);
    const [loading, setLoading] = useState(true);
    const [showModal, setShowModal] = useState(false);
    const [searchTerm, setSearchTerm] = useState('');
    const [filterCategoria, setFilterCategoria] = useState('');
    const [filterEstado, setFilterEstado] = useState('');
    const [submitting, setSubmitting] = useState(false);

    const [formData, setFormData] = useState({
        paciente_id: '',
        medico_id: '',
        fecha_emision: new Date().toISOString().split('T')[0],
        vigencia_dias: 30,
        instrucciones_generales: '',
        medicamentos: [{
            nombre_medicamento: '',
            dosis: '',
            via_administracion: 'oral',
            frecuencia: 'Cada 8 horas',
            duracion_dias: 7,
            cantidad_total: 21,
            instrucciones_especiales: ''
        }] as MedicamentoReceta[]
    });

    const [busquedaMedicamento, setBusquedaMedicamento] = useState('');
    const [medicamentosFiltrados, setMedicamentosFiltrados] = useState<Medicamento[]>([]);
    const [cargandoMedicamentos, setCargandoMedicamentos] = useState(false);
    const [indiceMedicamentoActual, setIndiceMedicamentoActual] = useState<number | null>(null);

    useEffect(() => {
        fetchRecetas();
        loadCatalogos();
    }, []);

    const fetchRecetas = async () => {
        try {
            const token = localStorage.getItem('token');
            const res = await axios.get('/api/recetas?per_page=50', {
                headers: { Authorization: `Bearer ${token}` }
            });
            if (res.data.success) {
                setRecetas(res.data.data);
            }
        } catch (error) {
            console.error('Error fetching recetas:', error);
        } finally {
            setLoading(false);
        }
    };

    const loadCatalogos = async () => {
        try {
            const token = localStorage.getItem('token');
            const headers = { Authorization: `Bearer ${token}` };

            // Cargar pacientes
            try {
                const resPacientes = await axios.get('/api/pacientes?per_page=100', { headers });
                if (resPacientes.data.success) {
                    setPacientes(resPacientes.data.data);
                }
            } catch (err) {
                console.error('Error loading pacientes:', err);
            }

            // Cargar medicos
            try {
                const resMedicos = await axios.get('/api/usuarios?per_page=100', { headers });
                if (resMedicos.data.success) {
                    setMedicos(resMedicos.data.data.filter((u: Medico) => u.rol === 'medico' || u.rol === 'admin'));
                }
            } catch (err) {
                console.error('Error loading medicos:', err);
            }
        } catch (error) {
            console.error('Error in loadCatalogos:', error);
        }
    };

    const buscarMedicamentos = async (query: string, indice: number) => {
        if (query.length < 2) {
            setMedicamentosFiltrados([]);
            return;
        }
        setCargandoMedicamentos(true);
        setIndiceMedicamentoActual(indice);
        try {
            const token = localStorage.getItem('token');
            const res = await axios.get(`/api/medicamentos/buscar?q=${encodeURIComponent(query)}`, {
                headers: { Authorization: `Bearer ${token}` }
            });
            setMedicamentosFiltrados(res.data.data || []);
        } catch (error) {
            console.error('Error searching medications:', error);
        } finally {
            setCargandoMedicamentos(false);
        }
    };

    const seleccionarMedicamento = (med: Medicamento, indice: number) => {
        const nuevosMedicamentos = [...formData.medicamentos];
        nuevosMedicamentos[indice] = {
            ...nuevosMedicamentos[indice],
            nombre_medicamento: med.nombre_generico,
            via_administracion: med.via_administracion
        };
        setFormData({ ...formData, medicamentos: nuevosMedicamentos });
        setMedicamentosFiltrados([]);
        setBusquedaMedicamento('');
        setIndiceMedicamentoActual(null);
    };

    const handleCreateReceta = async (e: React.FormEvent) => {
        e.preventDefault();
        setSubmitting(true);
        try {
            const token = localStorage.getItem('token');
            const payload = {
                paciente_id: parseInt(formData.paciente_id),
                medico_id: parseInt(formData.medico_id),
                fecha_emision: formData.fecha_emision,
                vigencia_dias: formData.vigencia_dias,
                instrucciones_generales: formData.instrucciones_generales,
                estado: 'activa',
                medicamentos: formData.medicamentos
            };

            const res = await axios.post('/api/recetas', payload, {
                headers: { Authorization: `Bearer ${token}` }
            });

            if (res.data.success) {
                alert('Receta creada exitosamente');
                setShowModal(false);
                resetForm();
                fetchRecetas();
            }
        } catch (error: any) {
            console.error(error);
            alert(error.response?.data?.message || 'Error al crear la receta');
        } finally {
            setSubmitting(false);
        }
    };

    const resetForm = () => {
        setFormData({
            paciente_id: '',
            medico_id: '',
            fecha_emision: new Date().toISOString().split('T')[0],
            vigencia_dias: 30,
            instrucciones_generales: '',
            medicamentos: [{
                nombre_medicamento: '',
                dosis: '',
                via_administracion: 'oral',
                frecuencia: 'Cada 8 horas',
                duracion_dias: 7,
                cantidad_total: 21,
                instrucciones_especiales: ''
            }]
        });
    };

    const addMedicamento = () => {
        setFormData({
            ...formData,
            medicamentos: [...formData.medicamentos, {
                nombre_medicamento: '',
                dosis: '',
                via_administracion: 'oral',
                frecuencia: 'Cada 8 horas',
                duracion_dias: 7,
                cantidad_total: 21,
                instrucciones_especiales: ''
            }]
        });
    };

    const removeMedicamento = (index: number) => {
        if (formData.medicamentos.length > 1) {
            const nuevos = formData.medicamentos.filter((_, i) => i !== index);
            setFormData({ ...formData, medicamentos: nuevos });
        }
    };

    const updateMedicamento = (index: number, field: keyof MedicamentoReceta, value: any) => {
        const nuevos = [...formData.medicamentos];
        nuevos[index] = { ...nuevos[index], [field]: value };
        setFormData({ ...formData, medicamentos: nuevos });
    };

    const filteredRecetas = recetas.filter(r => {
        if (searchTerm) {
            const term = searchTerm.toLowerCase();
            const matchPaciente = r.paciente?.nombre?.toLowerCase().includes(term) ||
                r.paciente?.apellido?.toLowerCase().includes(term);
            if (!matchPaciente) return false;
        }
        if (filterEstado && r.estado !== filterEstado) return false;
        return true;
    });

    const formatFecha = (dateString: string) => {
        if (!dateString) return '-';
        const date = new Date(dateString);
        return new Intl.DateTimeFormat('es-BO', { day: '2-digit', month: 'short', year: 'numeric' }).format(date);
    };

    return (
        <div className="space-y-6">
            <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                <div>
                    <h1 className={`text-2xl md:text-3xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'} flex items-center gap-2`}>
                        <Pill className="w-7 h-7 md:w-8 md:h-8 text-purple-500" />
                        Recetas Médicas
                    </h1>
                    <p className={`${isDarkMode ? 'text-gray-400' : 'text-gray-600'} mt-1 text-sm md:text-base`}>
                        Gestión de prescripciones y medicamentos
                    </p>
                </div>
                <button
                    onClick={() => setShowModal(true)}
                    className="px-4 py-2 bg-gradient-to-r from-purple-500 to-pink-500 text-white rounded-xl font-semibold shadow-lg hover:shadow-xl transform hover:scale-105 transition-all flex items-center gap-2"
                >
                    <Plus className="w-5 h-5" />
                    Nueva Receta
                </button>
            </div>

            <div className={`p-4 rounded-xl ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg`}>
                <div className="flex flex-col md:flex-row gap-4">
                    <div className="flex-1 relative">
                        <Search className={`absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`} />
                        <input
                            type="text"
                            placeholder="Buscar por paciente..."
                            value={searchTerm}
                            onChange={(e) => setSearchTerm(e.target.value)}
                            className={`w-full pl-10 pr-4 py-2 rounded-xl border-2 outline-none transition-all ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200 text-gray-700'}`}
                        />
                    </div>
                    <select
                        value={filterEstado}
                        onChange={(e) => setFilterEstado(e.target.value)}
                        className={`px-4 py-2 rounded-xl border-2 outline-none transition-all ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200 text-gray-700'}`}
                    >
                        <option value="">Todos los estados</option>
                        <option value="activa">Activa</option>
                        <option value="cancelada">Cancelada</option>
                        <option value="vencida">Vencida</option>
                    </select>
                </div>
            </div>

            <div className={`rounded-2xl overflow-hidden ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg`}>
                {loading ? (
                    <div className="p-12 text-center">
                        <div className="animate-spin rounded-full h-10 w-10 border-b-2 border-purple-500 mx-auto"></div>
                    </div>
                ) : filteredRecetas.length === 0 ? (
                    <div className={`p-12 text-center ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>
                        No hay recetas registradas.
                    </div>
                ) : (
                    <div className="overflow-x-auto">
                        <table className="w-full">
                            <thead className={`${isDarkMode ? 'bg-gray-900/50 text-gray-400' : 'bg-gray-50 text-gray-600'}`}>
                                <tr>
                                    <th className="px-6 py-4 text-left">Paciente</th>
                                    <th className="px-6 py-4 text-left">Médico</th>
                                    <th className="px-6 py-4 text-left">Fecha</th>
                                    <th className="px-6 py-4 text-left">Medicamentos</th>
                                    <th className="px-6 py-4 text-center">Estado</th>
                                    <th className="px-6 py-4 text-center">Acciones</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y dark:divide-gray-700">
                                {filteredRecetas.map((receta) => (
                                    <tr key={receta.id} className={`transition-colors ${isDarkMode ? 'hover:bg-gray-700/30' : 'hover:bg-gray-50'}`}>
                                        <td className="px-6 py-4">
                                            <div className={`font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                                                {receta.paciente?.nombre} {receta.paciente?.apellido}
                                            </div>
                                            <div className="text-xs text-gray-500">
                                                CI: {receta.paciente?.numero_documento}
                                            </div>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className="text-sm">
                                                Dr. {receta.medico?.nombre} {receta.medico?.apellido}
                                            </div>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className="text-sm font-mono">
                                                {formatFecha(receta.fecha_emision)}
                                            </div>
                                            <div className="text-xs text-gray-500">
                                                Vigencia: {receta.vigencia_dias} días
                                            </div>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className="text-sm">
                                                {receta.medicamentos?.length || 0} medicamento(s)
                                            </div>
                                            <div className="text-xs text-gray-500 max-w-xs truncate">
                                                {receta.medicamentos?.slice(0, 2).map(m => m.nombre_medicamento).join(', ')}
                                                {receta.medicamentos?.length > 2 && '...'}
                                            </div>
                                        </td>
                                        <td className="px-6 py-4 text-center">
                                            <span className={`px-3 py-1 text-xs font-bold rounded-full ${receta.estado === 'activa'
                                                    ? 'bg-green-100 text-green-800 dark:bg-green-900/40 dark:text-green-300'
                                                    : receta.estado === 'vencida'
                                                        ? 'bg-red-100 text-red-800 dark:bg-red-900/40 dark:text-red-300'
                                                        : 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300'
                                                }`}>
                                                {receta.estado?.toUpperCase()}
                                            </span>
                                        </td>
                                        <td className="px-6 py-4 text-center">
                                            <button className="p-2 text-blue-500 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg" title="Imprimir">
                                                <Printer className="w-4 h-4" />
                                            </button>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                )}
            </div>

            {showModal && (
                <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/60 backdrop-blur-sm p-4 overflow-y-auto">
                    <div className={`w-full max-w-4xl rounded-2xl shadow-2xl ${isDarkMode ? 'bg-gray-800 border border-gray-700' : 'bg-white'} my-8`}>
                        <div className="p-6 border-b dark:border-gray-700 flex justify-between items-center">
                            <h2 className={`text-xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                                📋 Nueva Receta Médica
                            </h2>
                            <button onClick={() => setShowModal(false)} className="text-gray-500 hover:text-gray-700">
                                <X className="w-6 h-6" />
                            </button>
                        </div>

                        <form onSubmit={handleCreateReceta} className="p-6 space-y-6 max-h-[70vh] overflow-y-auto">
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        <User className="w-4 h-4 inline mr-1" /> Paciente
                                    </label>
                                    <select
                                        required
                                        className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                        value={formData.paciente_id}
                                        onChange={e => setFormData({ ...formData, paciente_id: e.target.value })}
                                    >
                                        <option value="">Seleccione un paciente...</option>
                                        {pacientes.map(p => (
                                            <option key={p.id} value={p.id}>
                                                {p.nombre} {p.apellido} ({p.numero_documento})
                                            </option>
                                        ))}
                                    </select>
                                </div>
                                <div>
                                    <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        <FileText className="w-4 h-4 inline mr-1" /> Médico
                                    </label>
                                    <select
                                        required
                                        className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                        value={formData.medico_id}
                                        onChange={e => setFormData({ ...formData, medico_id: e.target.value })}
                                    >
                                        <option value="">Seleccione un médico...</option>
                                        {medicos.map(m => (
                                            <option key={m.id} value={m.id}>
                                                Dr. {m.nombre} {m.apellido}
                                            </option>
                                        ))}
                                    </select>
                                </div>
                            </div>

                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <div>
                                    <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        <Calendar className="w-4 h-4 inline mr-1" /> Fecha de Emisión
                                    </label>
                                    <input
                                        type="date"
                                        required
                                        className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                        value={formData.fecha_emision}
                                        onChange={e => setFormData({ ...formData, fecha_emision: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        Vigencia (días)
                                    </label>
                                    <input
                                        type="number"
                                        min="1"
                                        max="365"
                                        required
                                        className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                        value={formData.vigencia_dias}
                                        onChange={e => setFormData({ ...formData, vigencia_dias: parseInt(e.target.value) })}
                                    />
                                </div>
                            </div>

                            <div>
                                <label className={`block text-sm font-medium mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                    Instrucciones Generales
                                </label>
                                <textarea
                                    rows={2}
                                    className={`w-full p-2 rounded-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'}`}
                                    placeholder="Reposo relativo, dieta blanda, abundantes líquidos..."
                                    value={formData.instrucciones_generales}
                                    onChange={e => setFormData({ ...formData, instrucciones_generales: e.target.value })}
                                />
                            </div>

                            <div>
                                <div className="flex justify-between items-center mb-3">
                                    <label className={`text-sm font-bold ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                                        💊 Medicamentos
                                    </label>
                                    <button
                                        type="button"
                                        onClick={addMedicamento}
                                        className="text-sm text-purple-600 hover:text-purple-800 flex items-center gap-1"
                                    >
                                        <Plus className="w-4 h-4" /> Agregar
                                    </button>
                                </div>

                                <div className="space-y-3">
                                    {formData.medicamentos.map((med, index) => (
                                        <div key={index} className={`p-4 rounded-lg ${isDarkMode ? 'bg-gray-700/50 border border-gray-600' : 'bg-gray-50 border border-gray-200'}`}>
                                            <div className="flex justify-between items-start mb-2">
                                                <span className={`text-xs font-bold ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>
                                                    Medicamento #{index + 1}
                                                </span>
                                                {formData.medicamentos.length > 1 && (
                                                    <button
                                                        type="button"
                                                        onClick={() => removeMedicamento(index)}
                                                        className="text-red-500 hover:text-red-700"
                                                    >
                                                        <Trash2 className="w-4 h-4" />
                                                    </button>
                                                )}
                                            </div>

                                            <div className="grid grid-cols-1 md:grid-cols-12 gap-2">
                                                <div className="md:col-span-4 relative">
                                                    <input
                                                        type="text"
                                                        required
                                                        placeholder="Buscar medicamento..."
                                                        className={`w-full p-2 rounded border text-sm ${isDarkMode ? 'bg-gray-600 border-gray-500 text-white' : 'bg-white border-gray-300'}`}
                                                        value={index === indiceMedicamentoActual ? busquedaMedicamento : ''}
                                                        onChange={(e) => {
                                                            setBusquedaMedicamento(e.target.value);
                                                            buscarMedicamentos(e.target.value, index);
                                                        }}
                                                        onFocus={() => {
                                                            if (med.nombre_medicamento) {
                                                                setBusquedaMedicamento(med.nombre_medicamento);
                                                                buscarMedicamentos(med.nombre_medicamento, index);
                                                            }
                                                        }}
                                                    />
                                                    {indiceMedicamentoActual === index && medicamentosFiltrados.length > 0 && (
                                                        <div className={`absolute z-10 w-full mt-1 max-h-48 overflow-y-auto rounded-lg shadow-lg border ${isDarkMode ? 'bg-gray-700 border-gray-600' : 'bg-white border-gray-200'
                                                            }`}>
                                                            {medicamentosFiltrados.map((m) => (
                                                                <button
                                                                    key={m.id}
                                                                    type="button"
                                                                    className={`w-full text-left p-2 text-sm hover:bg-purple-100 dark:hover:bg-purple-900/30 ${isDarkMode ? 'text-gray-200' : 'text-gray-800'
                                                                        }`}
                                                                    onClick={() => seleccionarMedicamento(m, index)}
                                                                >
                                                                    <div className="font-medium">{m.nombre_generico}</div>
                                                                    <div className="text-xs text-gray-500">
                                                                        {m.nombre_comercial} | {m.concentracion} | Bs. {m.precio_bs}
                                                                    </div>
                                                                </button>
                                                            ))}
                                                        </div>
                                                    )}
                                                </div>
                                                <div className="md:col-span-2">
                                                    <input
                                                        type="text"
                                                        required
                                                        placeholder="Dosis"
                                                        className={`w-full p-2 rounded border text-sm ${isDarkMode ? 'bg-gray-600 border-gray-500 text-white' : 'bg-white border-gray-300'}`}
                                                        value={med.dosis}
                                                        onChange={(e) => updateMedicamento(index, 'dosis', e.target.value)}
                                                    />
                                                </div>
                                                <div className="md:col-span-2">
                                                    <select
                                                        required
                                                        className={`w-full p-2 rounded border text-sm ${isDarkMode ? 'bg-gray-600 border-gray-500 text-white' : 'bg-white border-gray-300'}`}
                                                        value={med.via_administracion}
                                                        onChange={(e) => updateMedicamento(index, 'via_administracion', e.target.value)}
                                                    >
                                                        {viasAdministracion.map(v => (
                                                            <option key={v.value} value={v.value}>{v.label}</option>
                                                        ))}
                                                    </select>
                                                </div>
                                                <div className="md:col-span-2">
                                                    <select
                                                        required
                                                        className={`w-full p-2 rounded border text-sm ${isDarkMode ? 'bg-gray-600 border-gray-500 text-white' : 'bg-white border-gray-300'}`}
                                                        value={med.frecuencia}
                                                        onChange={(e) => updateMedicamento(index, 'frecuencia', e.target.value)}
                                                    >
                                                        {frecuencias.map(f => (
                                                            <option key={f} value={f}>{f}</option>
                                                        ))}
                                                    </select>
                                                </div>
                                                <div className="md:col-span-1">
                                                    <input
                                                        type="number"
                                                        required
                                                        min="1"
                                                        placeholder="Días"
                                                        className={`w-full p-2 rounded border text-sm ${isDarkMode ? 'bg-gray-600 border-gray-500 text-white' : 'bg-white border-gray-300'}`}
                                                        value={med.duracion_dias}
                                                        onChange={(e) => updateMedicamento(index, 'duracion_dias', parseInt(e.target.value))}
                                                    />
                                                </div>
                                                <div className="md:col-span-1">
                                                    <input
                                                        type="number"
                                                        required
                                                        min="1"
                                                        placeholder="Cant."
                                                        className={`w-full p-2 rounded border text-sm ${isDarkMode ? 'bg-gray-600 border-gray-500 text-white' : 'bg-white border-gray-300'}`}
                                                        value={med.cantidad_total}
                                                        onChange={(e) => updateMedicamento(index, 'cantidad_total', parseInt(e.target.value))}
                                                    />
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </div>

                            <div className="flex justify-end gap-3 pt-4 border-t dark:border-gray-700">
                                <button
                                    type="button"
                                    onClick={() => setShowModal(false)}
                                    className={`px-4 py-2 rounded-lg font-medium ${isDarkMode ? 'text-gray-300 hover:bg-gray-700' : 'text-gray-600 hover:bg-gray-100'}`}
                                >
                                    Cancelar
                                </button>
                                <button
                                    type="submit"
                                    disabled={submitting}
                                    className="px-6 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded-lg font-semibold shadow-lg transition-colors disabled:opacity-50"
                                >
                                    {submitting ? 'Guardando...' : '💾 Guardar Receta'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
