import React, { useState, useEffect } from 'react';
import { useTheme } from '../../context/ThemeContext';
import { useUser } from '../../context/UserContext';
import axios from 'axios';

interface Usuario {
    id: number;
    nombre: string;
    apellido: string;
    email: string;
    telefono: string;
    rol: string;
    horario: string;
    estado: string;
}

export default function Usuarios() {
    const { theme } = useTheme();
    const { user } = useUser();
    const isDarkMode = theme === 'dark';
    const isAdmin = user?.rol === 'admin';

    const [usuarios, setUsuarios] = useState<Usuario[]>([]);
    const [loading, setLoading] = useState(true);

    const [buscar, setBuscar] = useState('');
    const [filtroRol, setFiltroRol] = useState('');
    const [mostrarModal, setMostrarModal] = useState(false);
    const [mostrarModalPassword, setMostrarModalPassword] = useState(false);
    const [usuarioEditando, setUsuarioEditando] = useState<Usuario | null>(null);
    const [usuarioCambiarPass, setUsuarioCambiarPass] = useState<Usuario | null>(null);
    const [guardando, setGuardando] = useState(false);
    const [mensaje, setMensaje] = useState({ tipo: '', texto: '' });
    const [passwordData, setPasswordData] = useState({ password: '', password_confirmation: '' });

    const [formData, setFormData] = useState({
        nombre: '',
        apellido: '',
        email: '',
        telefono: '',
        rol: 'recepcionista',
        horario: 'manana',
        password: '',
    });

    useEffect(() => {
        cargarUsuarios();
    }, [buscar, filtroRol]);

    const cargarUsuarios = async () => {
        setLoading(true);
        try {
            const params: any = { page: 1 };
            if (buscar) params.buscar = buscar;
            if (filtroRol) params.rol = filtroRol;

            const res = await axios.get('/usuarios', { params });
            setUsuarios(res.data.data);
        } catch (error) {
            console.error('Error:', error);
        } finally {
            setLoading(false);
        }
    };

    const abrirModalNuevo = () => {
        setUsuarioEditando(null);
        setFormData({ nombre: '', apellido: '', email: '', telefono: '', rol: 'recepcionista', horario: 'manana', password: '' });
        setMensaje({ tipo: '', texto: '' });
        setMostrarModal(true);
    };

    const abrirModalEditar = (usuario: Usuario) => {
        setUsuarioEditando(usuario);
        setFormData({ nombre: usuario.nombre, apellido: usuario.apellido, email: usuario.email, telefono: usuario.telefono || '', rol: usuario.rol, horario: usuario.horario || 'manana', password: '' });
        setMensaje({ tipo: '', texto: '' });
        setMostrarModal(true);
    };

    const guardarUsuario = async (e: React.FormEvent) => {
        e.preventDefault();
        setGuardando(true);
        setMensaje({ tipo: '', texto: '' });

        try {
            if (usuarioEditando) {
                const data: any = { nombre: formData.nombre, apellido: formData.apellido, rol: formData.rol, horario: formData.horario, telefono: formData.telefono };
                await axios.put(`/usuarios/${usuarioEditando.id}`, data);
                setMensaje({ tipo: 'success', texto: 'Usuario actualizado' });
            } else {
                await axios.post('/usuarios', formData);
                setMensaje({ tipo: 'success', texto: 'Usuario creado' });
            }
            cargarUsuarios();
            setTimeout(() => setMostrarModal(false), 1000);
        } catch (error: any) {
            setMensaje({ tipo: 'error', texto: error.response?.data?.message || 'Error' });
        } finally {
            setGuardando(false);
        }
    };

    const cambiarEstado = async (usuario: Usuario) => {
        const nuevoEstado = usuario.estado === 'activo' ? 'inactivo' : 'activo';
        try {
            await axios.patch(`/usuarios/${usuario.id}/cambiar-estado`, { estado: nuevoEstado });
            cargarUsuarios();
        } catch (error) { console.error(error); }
    };

    const eliminarUsuario = async (id: number) => {
        if (!confirm('¿Eliminar usuario?')) return;
        try { await axios.delete(`/usuarios/${id}`); cargarUsuarios(); } catch (error) { console.error(error); }
    };

    const abrirModalPassword = (usuario: Usuario) => {
        setUsuarioCambiarPass(usuario);
        setPasswordData({ password: '', password_confirmation: '' });
        setMensaje({ tipo: '', texto: '' });
        setMostrarModalPassword(true);
    };

    const cambiarPassword = async (e: React.FormEvent) => {
        e.preventDefault();
        if (passwordData.password !== passwordData.password_confirmation) {
            setMensaje({ tipo: 'error', texto: 'Las contraseñas no coinciden' });
            return;
        }
        setGuardando(true);
        try {
            await axios.patch(`/usuarios/${usuarioCambiarPass?.id}/password`, { password: passwordData.password, password_confirmation: passwordData.password_confirmation });
            setMensaje({ tipo: 'success', texto: 'Contraseña actualizada correctamente' });
            setTimeout(() => setMostrarModalPassword(false), 1500);
        } catch (error: any) {
            setMensaje({ tipo: 'error', texto: error.response?.data?.message || 'Error al cambiar contraseña' });
        } finally {
            setGuardando(false);
        }
    };

    const getRolLabel = (rol: string) => ({ admin: 'Admin', medico: 'Médico', enfermero: 'Enfermero', recepcionista: 'Recep' }[rol] || rol);

    return (
        <div className="space-y-6">
            <div>
                <h1 className={`text-3xl font-bold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>👥 Usuarios del Sistema</h1>
                <p className={`${isDarkMode ? 'text-gray-400' : 'text-gray-600'} mt-1`}>Gestión de personal</p>
            </div>

            <div className={`rounded-2xl p-6 ${isDarkMode ? 'bg-gray-800/50 border border-gray-700/50' : 'bg-white border border-gray-200'} shadow-lg`}>
                <div className="flex flex-col md:flex-row gap-4 items-center justify-between mb-6">
                    <div className="flex flex-col md:flex-row gap-3 items-center w-full md:w-auto">
                        <input type="text" placeholder="Buscar..." value={buscar} onChange={(e) => setBuscar(e.target.value)}
                            className={`w-full md:w-auto px-4 py-2 rounded-xl border-2 outline-none ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} />
                        <select value={filtroRol} onChange={(e) => setFiltroRol(e.target.value)}
                            className={`w-full md:w-auto px-4 py-2 rounded-xl border-2 outline-none ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`}>
                            <option value="">Todos</option>
                            <option value="admin">Admin</option>
                            <option value="medico">Médico</option>
                            <option value="enfermero">Enfermero</option>
                            <option value="recepcionista">Recepcionista</option>
                        </select>
                    </div>
                    {isAdmin && (
                        <button onClick={abrirModalNuevo}
                            className={`w-full md:w-auto px-6 py-2 rounded-xl font-bold ${isDarkMode ? 'bg-blue-600 text-white' : 'bg-blue-600 text-white'}`}>
                            ➕ Nuevo
                        </button>
                    )}
                </div>

                {loading ? <div className="text-center py-8"><div className={`animate-spin rounded-full h-12 w-12 border-b-2 mx-auto ${isDarkMode ? 'border-white' : 'border-blue-600'}`}></div></div> : (
                    <>
                        {/* Mobile Card View */}
                        <div className="md:hidden space-y-4">
                            {usuarios.length === 0 ? (
                                <div className={`text-center p-8 ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>Sin usuarios</div>
                            ) : (
                                usuarios.map((u) => (
                                    <div key={u.id} className={`p-4 rounded-xl ${isDarkMode ? 'bg-gray-800 border border-gray-700' : 'bg-white border border-gray-100'} shadow-sm`}>
                                        <div className="flex justify-between items-start mb-3">
                                            <div className="flex items-center gap-3">
                                                <div className={`w-10 h-10 rounded-full flex items-center justify-center font-bold ${isDarkMode ? 'bg-purple-600 text-white' : 'bg-blue-500 text-white'}`}>
                                                    {u.nombre.charAt(0)}
                                                </div>
                                                <div className="min-w-0 flex-1">
                                                    <p className={`font-semibold ${isDarkMode ? 'text-white' : 'text-gray-900'} truncate`}>{u.nombre} {u.apellido}</p>
                                                    <p className={`text-sm ${isDarkMode ? 'text-gray-500' : 'text-gray-500'} truncate`}>{u.email}</p>
                                                </div>
                                            </div>
                                            <span className={`px-2 py-1 rounded text-xs ${isDarkMode ? 'bg-purple-900 text-purple-400' : 'bg-purple-100 text-purple-700'}`}>{getRolLabel(u.rol)}</span>
                                        </div>

                                        <div className="space-y-2 mb-4">
                                            <div className="flex justify-between">
                                                <span className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>Teléfono:</span>
                                                <span className={`text-sm ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>{u.telefono || '-'}</span>
                                            </div>
                                            <div className="flex justify-between">
                                                <span className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>Turno:</span>
                                                <span className={`text-sm ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>{u.horario === 'manana' ? '🌅 Mañana' : '🌆 Tarde'}</span>
                                            </div>
                                            <div className="flex justify-between items-center">
                                                <span className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>Estado:</span>
                                                <button onClick={() => cambiarEstado(u)} className={`px-3 py-1 rounded text-xs ${u.estado === 'activo' ? (isDarkMode ? 'bg-green-900 text-green-400' : 'bg-green-100 text-green-700') : (isDarkMode ? 'bg-red-900 text-red-400' : 'bg-red-100 text-red-700')}`}>{u.estado}</button>
                                            </div>
                                        </div>

                                        <div className="flex justify-end gap-2 pt-3 border-t dark:border-gray-700">
                                            {isAdmin && (
                                                <>
                                                    <button onClick={() => abrirModalPassword(u)} className={`p-2 rounded ${isDarkMode ? 'hover:bg-gray-700 text-yellow-400' : 'hover:bg-gray-100 text-yellow-600'}`} title="Cambiar contraseña">🔑</button>
                                                    <button onClick={() => abrirModalEditar(u)} className={`p-2 rounded ${isDarkMode ? 'hover:bg-gray-700' : 'hover:bg-gray-100'}`}>✏️</button>
                                                    <button onClick={() => eliminarUsuario(u.id)} className={`p-2 rounded ${isDarkMode ? 'hover:bg-gray-700 text-red-400' : 'hover:bg-gray-100 text-red-600'}`}>🗑️</button>
                                                </>
                                            )}
                                        </div>
                                    </div>
                                ))
                            )}
                        </div>

                        {/* Desktop Table View */}
                        <div className="hidden md:block overflow-x-auto">
                            <table className="w-full">
                                <thead><tr className={isDarkMode ? 'text-gray-400' : 'text-gray-600'}><th className="text-left p-3">Usuario</th><th className="text-left p-3">Teléfono</th><th className="text-left p-3">Rol</th><th className="text-left p-3">Turno</th><th className="text-left p-3">Estado</th><th className="text-center p-3">Acciones</th></tr></thead>
                                <tbody>
                                    {usuarios.length === 0 ? <tr><td colSpan={6} className={`text-center p-8 ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>Sin usuarios</td></tr> :
                                        usuarios.map((u) => (
                                            <tr key={u.id} className={`border-t ${isDarkMode ? 'border-gray-700' : 'border-gray-200'}`}>
                                                <td className="p-3"><div className="flex items-center gap-3"><div className={`w-10 h-10 rounded-full flex items-center justify-center font-bold ${isDarkMode ? 'bg-purple-600 text-white' : 'bg-blue-500 text-white'}`}>{u.nombre.charAt(0)}</div><div><p className={`font-semibold ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>{u.nombre} {u.apellido}</p><p className={`text-sm ${isDarkMode ? 'text-gray-500' : 'text-gray-500'}`}>{u.email}</p></div></div></td>
                                                <td className={`p-3 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>{u.telefono || '-'}</td>
                                                <td className="p-3"><span className={`px-2 py-1 rounded text-xs ${isDarkMode ? 'bg-purple-900 text-purple-400' : 'bg-purple-100 text-purple-700'}`}>{getRolLabel(u.rol)}</span></td>
                                                <td className={`p-3 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>{u.horario === 'manana' ? '🌅 Mañana' : '🌆 Tarde'}</td>
                                                <td className="p-3"><button onClick={() => cambiarEstado(u)} className={`px-3 py-1 rounded text-xs ${u.estado === 'activo' ? (isDarkMode ? 'bg-green-900 text-green-400' : 'bg-green-100 text-green-700') : (isDarkMode ? 'bg-red-900 text-red-400' : 'bg-red-100 text-red-700')}`}>{u.estado}</button></td>
                                                <td className="p-3"><div className="flex justify-center gap-2">{isAdmin && (<><button onClick={() => abrirModalPassword(u)} className={`p-2 rounded ${isDarkMode ? 'hover:bg-gray-700 text-yellow-400' : 'hover:bg-gray-100 text-yellow-600'}`} title="Cambiar contraseña">🔑</button><button onClick={() => abrirModalEditar(u)} className={`p-2 rounded ${isDarkMode ? 'hover:bg-gray-700' : 'hover:bg-gray-100'}`}>✏️</button><button onClick={() => eliminarUsuario(u.id)} className={`p-2 rounded ${isDarkMode ? 'hover:bg-gray-700 text-red-400' : 'hover:bg-gray-100 text-red-600'}`}>🗑️</button></>)}</div></td>
                                            </tr>
                                        ))}
                                </tbody>
                            </table>
                        </div>
                    </>
                )}
            </div>

            {mostrarModal && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
                    <div className={`rounded-2xl p-6 w-full max-w-md ${isDarkMode ? 'bg-gray-800 border border-gray-700' : 'bg-white border border-gray-200'}`}>
                        <h2 className={`text-2xl font-bold mb-6 ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>{usuarioEditando ? '✏️ Editar' : '➕ Nuevo Usuario'}</h2>
                        {mensaje.texto && <div className={`mb-4 p-3 rounded ${mensaje.tipo === 'success' ? (isDarkMode ? 'bg-green-900 text-green-400' : 'bg-green-100 text-green-700') : (isDarkMode ? 'bg-red-900 text-red-400' : 'bg-red-100 text-red-700')}`}>{mensaje.texto}</div>}
                        <form onSubmit={guardarUsuario} className="space-y-4">
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div><label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Nombre</label><input type="text" value={formData.nombre} onChange={(e) => setFormData({ ...formData, nombre: e.target.value })} required className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} /></div>
                                <div><label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Apellido</label><input type="text" value={formData.apellido} onChange={(e) => setFormData({ ...formData, apellido: e.target.value })} required className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} /></div>
                            </div>
                            <div><label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Email</label><input type="email" value={formData.email} onChange={(e) => setFormData({ ...formData, email: e.target.value })} required={!usuarioEditando} disabled={!!usuarioEditando} className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'} ${usuarioEditando ? 'opacity-50' : ''}`} /></div>
                            <div><label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Teléfono</label><input type="text" value={formData.telefono} onChange={(e) => setFormData({ ...formData, telefono: e.target.value })} className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} /></div>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div><label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Rol</label><select value={formData.rol} onChange={(e) => setFormData({ ...formData, rol: e.target.value })} className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`}><option value="recepcionista">Recepcionista</option><option value="enfermero">Enfermero</option><option value="medico">Médico</option><option value="admin">Admin</option></select></div>
                                <div><label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Turno</label><select value={formData.horario} onChange={(e) => setFormData({ ...formData, horario: e.target.value })} className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`}><option value="manana">🌅 Mañana</option><option value="tarde">🌆 Tarde</option></select></div>
                            </div>
                            {!usuarioEditando && <div><label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Contraseña</label><input type="password" value={formData.password} onChange={(e) => setFormData({ ...formData, password: e.target.value })} required minLength={8} className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} /></div>}
                            <div className="flex gap-3 pt-4">
                                <button type="button" onClick={() => setMostrarModal(false)} className={`flex-1 p-2 rounded ${isDarkMode ? 'bg-gray-700 text-gray-300' : 'bg-gray-200 text-gray-700'}`}>Cancelar</button>
                                <button type="submit" disabled={guardando} className={`flex-1 p-2 rounded text-white ${isDarkMode ? 'bg-blue-600' : 'bg-blue-600'} disabled:opacity-50`}>{guardando ? '...' : usuarioEditando ? 'Actualizar' : 'Crear'}</button>
                            </div>
                        </form>
                    </div>
                </div>
            )}

            {mostrarModalPassword && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
                    <div className={`rounded-2xl p-6 w-full max-w-md ${isDarkMode ? 'bg-gray-800 border border-gray-700' : 'bg-white border border-gray-200'}`}>
                        <h2 className={`text-2xl font-bold mb-2 ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>🔑 Cambiar Contraseña</h2>
                        <p className={`mb-4 ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                            Usuario: <strong>{usuarioCambiarPass?.nombre} {usuarioCambiarPass?.apellido}</strong>
                        </p>
                        {mensaje.texto && <div className={`mb-4 p-3 rounded ${mensaje.tipo === 'success' ? (isDarkMode ? 'bg-green-900 text-green-400' : 'bg-green-100 text-green-700') : (isDarkMode ? 'bg-red-900 text-red-400' : 'bg-red-100 text-red-700')}`}>{mensaje.texto}</div>}
                        <form onSubmit={cambiarPassword} className="space-y-4">
                            <div><label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Nueva Contraseña</label><input type="password" value={passwordData.password} onChange={(e) => setPasswordData({ ...passwordData, password: e.target.value })} required minLength={8} className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} /></div>
                            <div><label className={`block text-sm mb-1 ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>Confirmar Contraseña</label><input type="password" value={passwordData.password_confirmation} onChange={(e) => setPasswordData({ ...passwordData, password_confirmation: e.target.value })} required className={`w-full p-2 rounded border ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-200'}`} /></div>
                            <div className="flex gap-3 pt-4">
                                <button type="button" onClick={() => setMostrarModalPassword(false)} className={`flex-1 p-2 rounded ${isDarkMode ? 'bg-gray-700 text-gray-300' : 'bg-gray-200 text-gray-700'}`}>Cancelar</button>
                                <button type="submit" disabled={guardando} className={`flex-1 p-2 rounded text-white ${isDarkMode ? 'bg-yellow-600' : 'bg-yellow-500'} disabled:opacity-50`}>{guardando ? 'Guardando...' : 'Cambiar'}</button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
