import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import LayoutPaciente from '../../components/portal-paciente/LayoutPaciente';
import { Loader2, Calendar, FlaskConical, AlertCircle, DollarSign, Eye } from 'lucide-react';

interface DashboardData {
    paciente_nombre: string;
    resumen: {
        total_citas: number;
        citas_proximas: number;
        total_examenes: number;
        examenes_pendientes_pago: number;
    };
    proximas_citas: any[];
    ultimos_examenes: any[];
}

const DashboardPaciente: React.FC = () => {
    const navigate = useNavigate();
    const [data, setData] = useState<DashboardData | null>(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState('');

    useEffect(() => {
        cargarDashboard();
    }, []);

    const cargarDashboard = async () => {
        try {
            const token = localStorage.getItem('paciente_token');
            const pacienteId = localStorage.getItem('paciente_id');

            if (!token || !pacienteId) {
                navigate('/paciente/login');
                return;
            }

            const response = await axios.get('/paciente/dashboard', {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'X-Paciente-Id': pacienteId
                }
            });

            if (response.data.success) {
                setData(response.data.data);
            } else {
                setError(response.data.message || 'Error al cargar datos');
            }
        } catch (err: any) {
            console.error('Error dashboard:', err);
            if (err.response?.status === 401) {
                localStorage.removeItem('paciente_token');
                localStorage.removeItem('paciente');
                localStorage.removeItem('paciente_id');
                navigate('/paciente/login');
            } else if (err.response?.status === 404) {
                // Si el endpoint no existe, mostrar datos vacíos
                setData({
                    paciente_nombre: '',
                    resumen: {
                        total_citas: 0,
                        citas_proximas: 0,
                        total_examenes: 0,
                        examenes_pendientes_pago: 0
                    },
                    proximas_citas: [],
                    ultimos_examenes: []
                });
            } else {
                setError(err.response?.data?.message || 'Error al cargar dashboard');
            }
        } finally {
            setLoading(false);
        }
    };

    if (loading) {
        return (
            <LayoutPaciente>
                <div className="flex items-center justify-center h-64">
                    <Loader2 className="animate-spin h-12 w-12 text-blue-600" />
                </div>
            </LayoutPaciente>
        );
    }

    if (error) {
        return (
            <LayoutPaciente>
                <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
                    {error}
                </div>
            </LayoutPaciente>
        );
    }

    if (!data) {
        return (
            <LayoutPaciente>
                <div className="bg-yellow-50 border border-yellow-200 text-yellow-700 px-4 py-3 rounded-lg">
                    No hay datos disponibles
                </div>
            </LayoutPaciente>
        );
    }

    return (
        <LayoutPaciente>
            <div className="space-y-8">
                {/* Welcome Section */}
                <div>
                    <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                        ¡Bienvenido, {data?.paciente_nombre?.split(' ')[0] || 'Paciente'}! 👋
                    </h1>
                    <p className="text-gray-600 dark:text-gray-400 mt-1">
                        Hoy es {new Date().toLocaleDateString('es-ES', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })}
                    </p>
                </div>

                {/* Acciones Rápidas */}
                <div>
                    <h2 className="text-lg font-bold text-gray-900 dark:text-white mb-4 flex items-center gap-2">
                        <span className="text-yellow-500">⚡</span> Acciones Rápidas
                    </h2>
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <button
                            onClick={() => navigate('/paciente/examenes')}
                            className="bg-gradient-to-br from-blue-500 to-cyan-500 rounded-xl p-6 text-white shadow-lg hover:shadow-xl hover:scale-[1.02] transition-all transform flex flex-col items-center justify-center group"
                        >
                            <div className="bg-white/20 p-3 rounded-full mb-3 group-hover:scale-110 transition-transform">
                                <FlaskConical className="w-8 h-8 text-white" />
                            </div>
                            <span className="font-bold text-lg">Ver Mis Exámenes</span>
                        </button>

                        <button
                            onClick={() => navigate('/paciente/perfil')}
                            className="bg-gradient-to-br from-green-500 to-emerald-500 rounded-xl p-6 text-white shadow-lg hover:shadow-xl hover:scale-[1.02] transition-all transform flex flex-col items-center justify-center group"
                        >
                            <div className="bg-white/20 p-3 rounded-full mb-3 group-hover:scale-110 transition-transform">
                                <Eye className="w-8 h-8 text-white" />
                            </div>
                            <span className="font-bold text-lg">Mi Perfil</span>
                        </button>

                        <button
                            className="bg-gradient-to-br from-purple-500 to-pink-500 rounded-xl p-6 text-white shadow-lg hover:shadow-xl hover:scale-[1.02] transition-all transform flex flex-col items-center justify-center group opacity-80 cursor-not-allowed"
                            title="Próximamente"
                        >
                            <div className="bg-white/20 p-3 rounded-full mb-3 group-hover:scale-110 transition-transform">
                                <Calendar className="w-8 h-8 text-white" />
                            </div>
                            <span className="font-bold text-lg">Nueva Cita</span>
                        </button>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    {/* Columna Izquierda: Citas y Exámenes Recientes */}
                    <div className="lg:col-span-2 space-y-8">
                        {/* Estadísticas Compactas */}
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                            <StatsCard
                                title="Total Citas"
                                value={data.resumen.total_citas}
                                icon={Calendar}
                                color="blue"
                            />
                            <StatsCard
                                title="Próximas"
                                value={data.resumen.citas_proximas}
                                icon={AlertCircle}
                                color="green"
                            />
                            <StatsCard
                                title="Exámenes"
                                value={data.resumen.total_examenes}
                                icon={FlaskConical}
                                color="purple"
                            />
                            <StatsCard
                                title="Pendientes"
                                value={data.resumen.examenes_pendientes_pago}
                                icon={DollarSign}
                                color="orange"
                            />
                        </div>

                        {/* Próximas Citas */}
                        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700 transition-colors">
                            <div className="p-6 border-b border-gray-100 dark:border-gray-700 flex justify-between items-center">
                                <h2 className="text-xl font-bold text-gray-900 dark:text-white flex items-center gap-2">
                                    <span className="text-blue-500">📅</span> Citas de Hoy
                                </h2>
                                <span className="text-xs font-medium text-blue-600 dark:text-blue-400 cursor-pointer hover:underline">Ver todas</span>
                            </div>
                            <div className="p-6">
                                {data.proximas_citas.length > 0 ? (
                                    <div className="space-y-4">
                                        {data.proximas_citas.map((cita) => (
                                            <div key={cita.id} className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-700/50 rounded-lg border border-gray-100 dark:border-gray-700">
                                                <div>
                                                    <p className="font-bold text-gray-900 dark:text-white">{cita.medico}</p>
                                                    <p className="text-sm text-gray-500 dark:text-gray-400">{cita.fecha_formateada} • {cita.especialidad}</p>
                                                </div>
                                                <span className="px-3 py-1 text-xs font-bold rounded-full bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300 uppercase tracking-wide">
                                                    {cita.estado}
                                                </span>
                                            </div>
                                        ))}
                                    </div>
                                ) : (
                                    <p className="text-gray-500 dark:text-gray-400 text-center py-4">No tienes citas programadas para hoy</p>
                                )}
                            </div>
                        </div>

                        {/* Últimos Exámenes */}
                        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700 transition-colors">
                            <div className="p-6 border-b border-gray-100 dark:border-gray-700 flex justify-between items-center">
                                <h2 className="text-xl font-bold text-gray-900 dark:text-white flex items-center gap-2">
                                    <span className="text-purple-500">🔬</span> Últimos Exámenes
                                </h2>
                                <button onClick={() => navigate('/paciente/examenes')} className="text-xs font-medium text-purple-600 dark:text-purple-400 cursor-pointer hover:underline">Ver todos</button>
                            </div>
                            <div className="p-6">
                                {data.ultimos_examenes.length > 0 ? (
                                    <div className="space-y-4">
                                        {data.ultimos_examenes.map((examen) => (
                                            <div key={examen.id} className="flex flex-col sm:flex-row sm:items-center justify-between p-4 bg-gray-50 dark:bg-gray-700/50 rounded-lg border border-gray-100 dark:border-gray-700 transition hover:bg-white dark:hover:bg-gray-700 hover:shadow-md">
                                                <div className="flex items-start gap-4">
                                                    <div className="p-2 bg-blue-100 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400 rounded-lg">
                                                        <FlaskConical className="w-5 h-5" />
                                                    </div>
                                                    <div>
                                                        <p className="font-bold text-gray-900 dark:text-white">{examen.tipo_examen}</p>
                                                        <p className="text-sm text-gray-500 dark:text-gray-400">{examen.fecha_formateada} • Dr. {examen.medico}</p>
                                                    </div>
                                                </div>
                                                <div className="mt-3 sm:mt-0 flex items-center gap-3">
                                                    {examen.estado_pago === 'pagado' ? (
                                                        <span className="px-3 py-1 text-xs font-bold rounded-full bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300">
                                                            PAGADO
                                                        </span>
                                                    ) : (
                                                        <span className="px-3 py-1 text-xs font-bold rounded-full bg-orange-100 dark:bg-orange-900/30 text-orange-700 dark:text-orange-300">
                                                            PENDIENTE
                                                        </span>
                                                    )}
                                                    {examen.puede_ver && (
                                                        <button
                                                            onClick={() => navigate('/paciente/examenes')}
                                                            className="p-2 text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition"
                                                            title="Ver Detalles"
                                                        >
                                                            <Eye className="w-5 h-5" />
                                                        </button>
                                                    )}
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                ) : (
                                    <p className="text-gray-500 dark:text-gray-400 text-center py-4">No hay actividad reciente</p>
                                )}
                            </div>
                        </div>
                    </div>

                    {/* Columna Derecha: Resumen */}
                    <div className="space-y-8">
                        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700 transition-colors h-full">
                            <div className="p-6 border-b border-gray-100 dark:border-gray-700">
                                <h2 className="text-lg font-bold text-gray-900 dark:text-white flex items-center gap-2">
                                    📊 Resumen de Actividad
                                </h2>
                            </div>
                            <div className="p-6 space-y-6">
                                <div className="flex items-start gap-4 p-4 rounded-xl bg-blue-50 dark:bg-blue-900/20">
                                    <div className="p-3 bg-blue-500 rounded-lg text-white shadow-lg shadow-blue-500/30">
                                        <Calendar className="w-6 h-6" />
                                    </div>
                                    <div>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">{data.resumen.total_citas}</p>
                                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Citas Totales</p>
                                        <p className="text-xs text-blue-500 mt-1">Histórico</p>
                                    </div>
                                </div>

                                <div className="flex items-start gap-4 p-4 rounded-xl bg-purple-50 dark:bg-purple-900/20">
                                    <div className="p-3 bg-purple-500 rounded-lg text-white shadow-lg shadow-purple-500/30">
                                        <FlaskConical className="w-6 h-6" />
                                    </div>
                                    <div>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">{data.resumen.total_examenes}</p>
                                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Exámenes Realizados</p>
                                        <p className="text-xs text-purple-500 mt-1">Este año</p>
                                    </div>
                                </div>

                                <div className="flex items-start gap-4 p-4 rounded-xl bg-orange-50 dark:bg-orange-900/20">
                                    <div className="p-3 bg-orange-500 rounded-lg text-white shadow-lg shadow-orange-500/30">
                                        <DollarSign className="w-6 h-6" />
                                    </div>
                                    <div>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">{data.resumen.examenes_pendientes_pago}</p>
                                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Pagos Pendientes</p>
                                        <p className="text-xs text-orange-500 mt-1">Requiere atención</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </LayoutPaciente>
    );
};

// Componente de tarjeta de estadísticas (Updated Style)
const StatsCard: React.FC<{
    title: string;
    value: number;
    icon: React.ElementType;
    color: string;
}> = ({ title, value, icon: Icon, color }) => {
    const colorClasses = {
        blue: { bg: 'bg-blue-100 dark:bg-blue-900/30', text: 'text-blue-600 dark:text-blue-400' },
        green: { bg: 'bg-green-100 dark:bg-green-900/30', text: 'text-green-600 dark:text-green-400' },
        purple: { bg: 'bg-purple-100 dark:bg-purple-900/30', text: 'text-purple-600 dark:text-purple-400' },
        orange: { bg: 'bg-orange-100 dark:bg-orange-900/30', text: 'text-orange-600 dark:text-orange-400' },
    };

    const currentColors = colorClasses[color as keyof typeof colorClasses];

    return (
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700 p-4 flex items-center gap-4 transition-transform hover:scale-105">
            <div className={`p-3 rounded-lg ${currentColors.bg} ${currentColors.text}`}>
                <Icon className="w-6 h-6" />
            </div>
            <div>
                <p className="text-2xl font-bold text-gray-900 dark:text-white">{value}</p>
                <p className="text-xs font-medium text-gray-500 dark:text-gray-400">{title}</p>
            </div>
        </div>
    );
};

export default DashboardPaciente;
