import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import LayoutPaciente from '../../components/portal-paciente/LayoutPaciente';
import { Loader2, ChevronDown, ChevronUp, Pill, FlaskConical, HeartPulse, Stethoscope } from 'lucide-react';

interface Medicamento { nombre: string; dosis: string; frecuencia: string; duracion: string; }
interface Receta { id: number; fecha_emision: string; instrucciones_generales?: string; medicamentos: Medicamento[]; }
interface Examen { id: number; descripcion: string; tipo: string; estado: string; resultados?: string; fecha: string; }
interface Historia {
    id: number;
    fecha_formateada: string;
    motivo_consulta: string;
    diagnostico: string;
    tratamiento?: string;
    recomendaciones?: string;
    proxima_cita?: string;
    presion_arterial?: string;
    frecuencia_cardiaca?: number;
    temperatura?: number;
    peso?: number;
    altura?: number;
    saturacion_oxigeno?: number;
    notas_soap?: { subjetivo?: string; objetivo?: string; analisis?: string; plan?: string; };
    medico?: { nombre: string; especialidad: string; };
    cie10?: string;
    tipo_diagnostico?: string;
    examenes: Examen[];
    recetas: Receta[];
}

const HistorialPaciente: React.FC = () => {
    const navigate = useNavigate();
    const [historias, setHistorias] = useState<Historia[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState('');
    const [expandedId, setExpandedId] = useState<number | null>(null);

    useEffect(() => { cargarHistorial(); }, []);

    const cargarHistorial = async () => {
        try {
            const token = localStorage.getItem('paciente_token');
            const pacienteId = localStorage.getItem('paciente_id');
            if (!token || !pacienteId) { navigate('/paciente/login'); return; }

            const res = await axios.get('/paciente/historias', {
                headers: { 'Authorization': `Bearer ${token}`, 'X-Paciente-Id': pacienteId }
            });
            if (res.data.success) setHistorias(res.data.data);
        } catch (err: any) {
            if (err.response?.status === 401) { navigate('/paciente/login'); }
            else setError(err.response?.data?.message || 'Error al cargar el historial clínico');
        } finally {
            setLoading(false);
        }
    };

    const toggle = (id: number) => setExpandedId(expandedId === id ? null : id);

    const estadoExamen = (e: string) => {
        const map: Record<string, string> = { completado: '✅ Completado', solicitado: '⏳ Solicitado', en_proceso: '🔄 En proceso', cancelado: '❌ Cancelado' };
        return map[e] ?? e;
    };

    if (loading) return (
        <LayoutPaciente>
            <div className="flex items-center justify-center h-64">
                <Loader2 className="animate-spin h-12 w-12 text-blue-600" />
            </div>
        </LayoutPaciente>
    );

    return (
        <LayoutPaciente>
            <div className="space-y-6">
                <div>
                    <h1 className="text-2xl font-bold text-gray-900 dark:text-white flex items-center gap-2">
                        <Stethoscope className="w-7 h-7 text-blue-600" /> Mi Historial Clínico
                    </h1>
                    <p className="text-gray-500 dark:text-gray-400 mt-1">Registro completo de todas tus consultas médicas</p>
                </div>

                {error && (
                    <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">{error}</div>
                )}

                {historias.length === 0 && !error ? (
                    <div className="text-center py-20 text-gray-400 dark:text-gray-500">
                        <Stethoscope className="w-16 h-16 mx-auto mb-4 opacity-30" />
                        <p className="text-lg font-medium">Sin consultas registradas</p>
                    </div>
                ) : (
                    <div className="space-y-4">
                        {historias.map((h, idx) => (
                            <div key={h.id} className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700 overflow-hidden transition-all">
                                {/* Encabezado */}
                                <button
                                    onClick={() => toggle(h.id)}
                                    className="w-full flex items-center justify-between p-5 hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors text-left"
                                >
                                    <div className="flex items-center gap-4">
                                        <div className={`w-3 h-3 rounded-full flex-shrink-0 ${idx === historias.length - 1 ? 'bg-purple-500' : 'bg-blue-500'}`} />
                                        <div>
                                            <div className="flex items-center gap-2 flex-wrap">
                                                <span className="font-bold text-gray-900 dark:text-white">{h.motivo_consulta}</span>
                                                <span className={`text-xs px-2 py-0.5 rounded-full font-semibold ${idx === historias.length - 1 ? 'bg-purple-100 text-purple-700' : 'bg-blue-100 text-blue-700'}`}>
                                                    {idx === historias.length - 1 ? '🌟 Primera Consulta' : '🔄 Seguimiento'}
                                                </span>
                                            </div>
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mt-0.5">
                                                📅 {h.fecha_formateada} &bull; {h.medico?.nombre} &bull; {h.medico?.especialidad}
                                            </p>
                                            {h.cie10 && (
                                                <p className="text-xs mt-1 text-red-600 dark:text-red-400 font-medium">{h.cie10}</p>
                                            )}
                                        </div>
                                    </div>
                                    {expandedId === h.id ? <ChevronUp className="w-5 h-5 text-gray-400 flex-shrink-0" /> : <ChevronDown className="w-5 h-5 text-gray-400 flex-shrink-0" />}
                                </button>

                                {/* Detalle expandible */}
                                {expandedId === h.id && (
                                    <div className="px-5 pb-6 border-t border-gray-100 dark:border-gray-700 space-y-5 pt-4">

                                        {/* Signos vitales */}
                                        {(h.presion_arterial || h.peso || h.temperatura) && (
                                            <div>
                                                <h3 className="text-xs font-bold uppercase tracking-wider text-gray-500 dark:text-gray-400 mb-3 flex items-center gap-1">
                                                    <HeartPulse className="w-4 h-4" /> Signos Vitales
                                                </h3>
                                                <div className="grid grid-cols-2 sm:grid-cols-4 gap-3">
                                                    {h.presion_arterial && <VitalCard label="Presión Arterial" value={h.presion_arterial} unit="mmHg" />}
                                                    {h.frecuencia_cardiaca && <VitalCard label="Frecuencia Cardíaca" value={String(h.frecuencia_cardiaca)} unit="lpm" />}
                                                    {h.temperatura && <VitalCard label="Temperatura" value={String(h.temperatura)} unit="°C" />}
                                                    {h.peso && <VitalCard label="Peso" value={String(h.peso)} unit="kg" />}
                                                    {h.altura && <VitalCard label="Talla" value={String(h.altura)} unit="m" />}
                                                    {h.saturacion_oxigeno && <VitalCard label="SpO₂" value={String(h.saturacion_oxigeno)} unit="%" />}
                                                </div>
                                            </div>
                                        )}

                                        {/* SOAP */}
                                        {h.notas_soap && (
                                            <div>
                                                <h3 className="text-xs font-bold uppercase tracking-wider text-gray-500 dark:text-gray-400 mb-3">Notas SOAP</h3>
                                                <div className="space-y-2">
                                                    {h.notas_soap.subjetivo && <SoapBlock color="blue" label="S – Subjetivo" text={h.notas_soap.subjetivo} />}
                                                    {h.notas_soap.objetivo && <SoapBlock color="green" label="O – Objetivo" text={h.notas_soap.objetivo} />}
                                                    {h.notas_soap.analisis && <SoapBlock color="orange" label="A – Análisis" text={h.notas_soap.analisis} />}
                                                    {h.notas_soap.plan && <SoapBlock color="purple" label="P – Plan" text={h.notas_soap.plan} />}
                                                </div>
                                            </div>
                                        )}

                                        {/* Diagnóstico / Tratamiento */}
                                        {h.diagnostico && (
                                            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                                <div className="bg-red-50 dark:bg-red-900/10 border-l-4 border-red-400 p-3 rounded-r-lg">
                                                    <p className="text-xs font-bold text-red-700 dark:text-red-400 mb-1">Diagnóstico</p>
                                                    <p className="text-sm text-gray-700 dark:text-gray-300">{h.diagnostico}</p>
                                                </div>
                                                {h.tratamiento && (
                                                    <div className="bg-teal-50 dark:bg-teal-900/10 border-l-4 border-teal-400 p-3 rounded-r-lg">
                                                        <p className="text-xs font-bold text-teal-700 dark:text-teal-400 mb-1">Tratamiento</p>
                                                        <p className="text-sm text-gray-700 dark:text-gray-300">{h.tratamiento}</p>
                                                    </div>
                                                )}
                                            </div>
                                        )}

                                        {/* Recomendaciones */}
                                        {h.recomendaciones && (
                                            <div className="bg-yellow-50 dark:bg-yellow-900/10 border-l-4 border-yellow-400 p-3 rounded-r-lg">
                                                <p className="text-xs font-bold text-yellow-700 dark:text-yellow-400 mb-1">Recomendaciones</p>
                                                <p className="text-sm text-gray-700 dark:text-gray-300">{h.recomendaciones}</p>
                                            </div>
                                        )}

                                        {/* Recetas */}
                                        {h.recetas.length > 0 && (
                                            <div>
                                                <h3 className="text-xs font-bold uppercase tracking-wider text-gray-500 dark:text-gray-400 mb-3 flex items-center gap-1">
                                                    <Pill className="w-4 h-4" /> Receta Médica
                                                </h3>
                                                {h.recetas.map(r => (
                                                    <div key={r.id} className="bg-green-50 dark:bg-green-900/10 rounded-lg p-3 border border-green-200 dark:border-green-800 mb-2">
                                                        {r.instrucciones_generales && (
                                                            <p className="text-xs text-green-700 dark:text-green-400 mb-2 italic">{r.instrucciones_generales}</p>
                                                        )}
                                                        <div className="overflow-x-auto">
                                                            <table className="w-full text-xs">
                                                                <thead>
                                                                    <tr className="text-gray-500 border-b border-green-200 dark:border-green-800">
                                                                        <th className="text-left pb-1">Medicamento</th>
                                                                        <th className="text-left pb-1">Dosis</th>
                                                                        <th className="text-left pb-1">Frecuencia</th>
                                                                        <th className="text-left pb-1">Duración</th>
                                                                    </tr>
                                                                </thead>
                                                                <tbody>
                                                                    {r.medicamentos.map((m, i) => (
                                                                        <tr key={i} className="border-b border-green-100 dark:border-green-900/30 last:border-0">
                                                                            <td className="py-1 font-semibold text-gray-900 dark:text-white">{m.nombre}</td>
                                                                            <td className="py-1 text-gray-600 dark:text-gray-300">{m.dosis}</td>
                                                                            <td className="py-1 text-gray-600 dark:text-gray-300">{m.frecuencia}</td>
                                                                            <td className="py-1 text-gray-600 dark:text-gray-300">{m.duracion}</td>
                                                                        </tr>
                                                                    ))}
                                                                </tbody>
                                                            </table>
                                                        </div>
                                                    </div>
                                                ))}
                                            </div>
                                        )}

                                        {/* Exámenes */}
                                        {h.examenes.length > 0 && (
                                            <div>
                                                <h3 className="text-xs font-bold uppercase tracking-wider text-gray-500 dark:text-gray-400 mb-3 flex items-center gap-1">
                                                    <FlaskConical className="w-4 h-4" /> Exámenes Solicitados
                                                </h3>
                                                <div className="space-y-2">
                                                    {h.examenes.map(e => (
                                                        <div key={e.id} className="bg-blue-50 dark:bg-blue-900/10 rounded-lg p-3 border border-blue-200 dark:border-blue-800">
                                                            <div className="flex justify-between items-start">
                                                                <div>
                                                                    <p className="font-semibold text-sm text-gray-900 dark:text-white">{e.descripcion}</p>
                                                                    <p className="text-xs text-gray-500 mt-0.5">{e.tipo} • {e.fecha}</p>
                                                                </div>
                                                                <span className="text-xs">{estadoExamen(e.estado)}</span>
                                                            </div>
                                                            {e.resultados && (
                                                                <div className="mt-2 pt-2 border-t border-blue-200 dark:border-blue-700">
                                                                    <p className="text-xs font-bold text-blue-700 dark:text-blue-300 mb-1">Resultados:</p>
                                                                    <p className="text-xs text-gray-600 dark:text-gray-300 leading-relaxed">{e.resultados}</p>
                                                                </div>
                                                            )}
                                                        </div>
                                                    ))}
                                                </div>
                                            </div>
                                        )}

                                        {/* Próxima cita */}
                                        {h.proxima_cita && (
                                            <div className="flex items-center gap-2 text-sm text-indigo-600 dark:text-indigo-400 bg-indigo-50 dark:bg-indigo-900/20 p-3 rounded-lg">
                                                <span>📅</span>
                                                <span>Próxima cita programada: <strong>{new Date(h.proxima_cita).toLocaleDateString('es-ES', { day: '2-digit', month: 'long', year: 'numeric' })}</strong></span>
                                            </div>
                                        )}
                                    </div>
                                )}
                            </div>
                        ))}
                    </div>
                )}
            </div>
        </LayoutPaciente>
    );
};

const VitalCard: React.FC<{ label: string; value: string; unit: string }> = ({ label, value, unit }) => (
    <div className="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-3 text-center">
        <p className="text-xs text-gray-500 dark:text-gray-400 mb-1">{label}</p>
        <p className="text-lg font-bold text-gray-900 dark:text-white">{value}<span className="text-xs font-normal text-gray-400 ml-1">{unit}</span></p>
    </div>
);

const colorMap: Record<string, string> = {
    blue: 'border-blue-400 bg-blue-50 dark:bg-blue-900/10 text-blue-800 dark:text-blue-300',
    green: 'border-green-400 bg-green-50 dark:bg-green-900/10 text-green-800 dark:text-green-300',
    orange: 'border-orange-400 bg-orange-50 dark:bg-orange-900/10 text-orange-800 dark:text-orange-300',
    purple: 'border-purple-400 bg-purple-50 dark:bg-purple-900/10 text-purple-800 dark:text-purple-300',
};

const SoapBlock: React.FC<{ label: string; text: string; color: string }> = ({ label, text, color }) => (
    <div className={`border-l-4 p-3 rounded-r-lg ${colorMap[color]}`}>
        <p className="text-xs font-bold mb-1 opacity-70">{label}</p>
        <p className="text-sm leading-relaxed opacity-90">{text}</p>
    </div>
);

export default HistorialPaciente;
