import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import { Sun, Moon, User, Smartphone, Loader2, FileText } from 'lucide-react';

const LoginPaciente: React.FC = () => {
    const navigate = useNavigate();
    const [ci, setCi] = useState('');
    const [celular, setCelular] = useState('');
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState('');
    const [isDarkMode, setIsDarkMode] = useState(false);

    useEffect(() => {
        // Cargar preferencia de modo oscuro
        const savedMode = localStorage.getItem('paciente_dark_mode');
        if (savedMode) {
            setIsDarkMode(savedMode === 'true');
        } else {
            const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
            setIsDarkMode(prefersDark);
        }
    }, []);

    const toggleDarkMode = () => {
        const newMode = !isDarkMode;
        setIsDarkMode(newMode);
        localStorage.setItem('paciente_dark_mode', newMode.toString());
    };

    const handleLogin = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');
        setLoading(true);

        try {
            const response = await axios.post('/paciente/login', {
                ci,
                celular,
            });

            if (response.data.success) {
                // Guardar token y datos del paciente
                localStorage.setItem('paciente_token', response.data.token);
                localStorage.setItem('paciente', JSON.stringify(response.data.paciente));
                localStorage.setItem('paciente_id', response.data.paciente.id.toString());

                // Redirigir al dashboard del paciente
                navigate('/paciente/dashboard');
            }
        } catch (err: any) {
            setError(err.response?.data?.message || 'Error al iniciar sesión. Verifica tus credenciales.');
            console.error('Error login:', err);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className={`min-h-screen flex flex-col items-center justify-center px-4 transition-colors duration-300 ${isDarkMode ? 'bg-gray-900' : 'bg-gradient-to-br from-blue-500 via-cyan-500 to-teal-500'}`}>
            {/* Toggle Modo Oscuro */}
            <button
                onClick={toggleDarkMode}
                className={`absolute top-4 right-4 p-3 rounded-full transition-colors ${isDarkMode ? 'bg-gray-800 text-yellow-400 hover:bg-gray-700' : 'bg-white/20 text-white hover:bg-white/30'}`}
                title={isDarkMode ? 'Cambiar a modo claro' : 'Cambiar a modo oscuro'}
            >
                {isDarkMode ? <Sun className="w-6 h-6" /> : <Moon className="w-6 h-6" />}
            </button>

            {/* Contenedor principal */}
            <div className={`rounded-2xl shadow-2xl p-8 w-full max-w-md transition-colors duration-300 ${isDarkMode ? 'bg-gray-800 border border-gray-700' : 'bg-white'}`}>
                {/* Logo */}
                <div className="text-center mb-8">
                    <div className="inline-block p-4 bg-gradient-to-br from-blue-600 to-cyan-600 rounded-full mb-4 shadow-lg">
                        <FileText className="w-12 h-12 text-white" />
                    </div>
                    <h1 className={`text-3xl font-bold transition-colors ${isDarkMode ? 'text-white' : 'text-gray-800'}`}>Portal del Paciente</h1>
                    <p className={`mt-2 transition-colors ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>Cibertronia Salud</p>
                </div>

                {/* Formulario */}
                <form onSubmit={handleLogin} className="space-y-6">
                    {error && (
                        <div className={`border px-4 py-3 rounded-lg ${isDarkMode ? 'bg-red-900/30 border-red-700 text-red-300' : 'bg-red-50 border-red-200 text-red-700'}`}>
                            <p className="text-sm">{error}</p>
                        </div>
                    )}

                    {/* Campo C.I. */}
                    <div>
                        <label htmlFor="ci" className={`block text-sm font-medium mb-2 transition-colors ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                            Cédula de Identidad (C.I.)
                        </label>
                        <div className="relative">
                            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <User className={`h-5 w-5 ${isDarkMode ? 'text-gray-500' : 'text-gray-400'}`} />
                            </div>
                            <input
                                type="text"
                                id="ci"
                                value={ci}
                                onChange={(e) => setCi(e.target.value)}
                                required
                                className={`block w-full pl-10 pr-3 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors ${
                                    isDarkMode 
                                        ? 'bg-gray-700 border-gray-600 text-white placeholder-gray-400' 
                                        : 'bg-white border-gray-300 text-gray-900 placeholder-gray-400'
                                }`}
                                placeholder="Ej: 12345678"
                            />
                        </div>
                    </div>

                    {/* Campo Celular */}
                    <div>
                        <label htmlFor="celular" className={`block text-sm font-medium mb-2 transition-colors ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>
                            Número de Celular
                        </label>
                        <div className="relative">
                            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <Smartphone className={`h-5 w-5 ${isDarkMode ? 'text-gray-500' : 'text-gray-400'}`} />
                            </div>
                            <input
                                type="password"
                                id="celular"
                                value={celular}
                                onChange={(e) => setCelular(e.target.value)}
                                required
                                className={`block w-full pl-10 pr-3 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors ${
                                    isDarkMode 
                                        ? 'bg-gray-700 border-gray-600 text-white placeholder-gray-400' 
                                        : 'bg-white border-gray-300 text-gray-900 placeholder-gray-400'
                                }`}
                                placeholder="••••••••"
                            />
                        </div>
                        <p className={`text-xs mt-1 ${isDarkMode ? 'text-gray-500' : 'text-gray-500'}`}>
                            Usa tu número de celular registrado como contraseña
                        </p>
                    </div>

                    <button
                        type="submit"
                        disabled={loading}
                        className="w-full bg-gradient-to-r from-blue-600 to-cyan-600 text-white font-semibold py-3 px-4 rounded-lg hover:from-blue-700 hover:to-cyan-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed shadow-lg flex items-center justify-center"
                    >
                        {loading ? (
                            <>
                                <Loader2 className="animate-spin -ml-1 mr-3 h-5 w-5" />
                                Iniciando sesión...
                            </>
                        ) : (
                            'Ingresar'
                        )}
                    </button>
                </form>

                {/* Información adicional */}
                <div className="mt-6 text-center">
                    <p className={`text-sm transition-colors ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                        Ingrese su C.I. y número de celular registrado
                    </p>
                </div>

                {/* Instrucciones */}
                <div className={`mt-6 border rounded-lg p-4 ${isDarkMode ? 'bg-blue-900/20 border-blue-800' : 'bg-blue-50 border-blue-200'}`}>
                    <p className={`text-xs font-semibold mb-2 ${isDarkMode ? 'text-blue-400' : 'text-blue-800'}`}>🔐 Cómo acceder:</p>
                    <p className={`text-xs mb-1 ${isDarkMode ? 'text-blue-300' : 'text-blue-700'}`}>• Usuario: Tu Cédula de Identidad</p>
                    <p className={`text-xs ${isDarkMode ? 'text-blue-300' : 'text-blue-700'}`}>• Contraseña: Tu número de celular registrado</p>
                </div>
            </div>

            {/* Footer */}
            <div className={`mt-8 text-center ${isDarkMode ? 'text-gray-500' : 'text-white/80'}`}>
                <p className="text-sm">© 2024 Cibertronia Salud. Todos los derechos reservados.</p>
                <a href="/" className="text-sm hover:underline mt-2 inline-block">← Volver al inicio</a>
            </div>
        </div>
    );
};

export default LoginPaciente;
