import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import LayoutPaciente from '../../components/portal-paciente/LayoutPaciente';
import { Loader2, Eye, Download, Calendar, User, Building2, CreditCard, DollarSign, X, FileText } from 'lucide-react';

interface Examen {
    id: number;
    tipo_examen: string;
    descripcion: string;
    fecha_formateada: string;
    medico_solicitante: string;
    laboratorio_centro: string;
    estado_examen_formateado: string;
    estado_pago: string;
    estado_pago_formateado: string;
    monto: number;
    metodo_pago: string;
    fecha_pago: string;
    tiene_resultado: boolean;
    puede_ver_resultado: boolean;
    archivo_resultado: string;
}

const MisExamenes: React.FC = () => {
    const navigate = useNavigate();
    const [examenes, setExamenes] = useState<Examen[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState('');
    const [filtro, setFiltro] = useState('todos');

    // Estado para el modal del visor PDF
    const [pdfModalOpen, setPdfModalOpen] = useState(false);
    const [pdfUrl, setPdfUrl] = useState<string | null>(null);
    const [pdfLoading, setPdfLoading] = useState(false);
    const [selectedExamen, setSelectedExamen] = useState<Examen | null>(null);

    useEffect(() => {
        cargarExamenes();
    }, [filtro]);

    const cargarExamenes = async () => {
        try {
            const token = localStorage.getItem('paciente_token');
            const pacienteId = localStorage.getItem('paciente_id');

            if (!token || !pacienteId) {
                navigate('/paciente/login');
                return;
            }

            const response = await axios.get(`/paciente/examenes?estado=${filtro}`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'X-Paciente-Id': pacienteId
                }
            });

            if (response.data.success) {
                setExamenes(response.data.data);
            }
        } catch (err: any) {
            if (err.response?.status === 401) {
                localStorage.removeItem('paciente_token');
                localStorage.removeItem('paciente_id');
                navigate('/paciente/login');
            } else {
                setError('Error al cargar exámenes');
            }
        } finally {
            setLoading(false);
        }
    };

    const abrirVisorPDF = async (examen: Examen) => {
        setSelectedExamen(examen);
        setPdfLoading(true);
        setPdfModalOpen(true);

        try {
            const token = localStorage.getItem('paciente_token');
            const pacienteId = localStorage.getItem('paciente_id');
            const url = `/paciente/examenes/${examen.id}/ver?token=${token}&paciente_id=${pacienteId}`;
            setPdfUrl(url);
        } catch (err) {
            console.error('Error al cargar PDF:', err);
        } finally {
            setPdfLoading(false);
        }
    };

    const cerrarVisorPDF = () => {
        setPdfModalOpen(false);
        setPdfUrl(null);
        setSelectedExamen(null);
    };

    const descargarExamen = async (id: number, tipoExamen: string) => {
        try {
            const token = localStorage.getItem('paciente_token');
            const pacienteId = localStorage.getItem('paciente_id');

            const response = await axios.get(`/paciente/examenes/${id}/descargar`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'X-Paciente-Id': pacienteId
                },
                responseType: 'blob',
            });

            // Crear URL del blob y descargar
            const url = window.URL.createObjectURL(new Blob([response.data]));
            const link = document.createElement('a');
            link.href = url;
            link.setAttribute('download', `${tipoExamen}_resultado.pdf`);
            document.body.appendChild(link);
            link.click();
            link.remove();
            window.URL.revokeObjectURL(url);
        } catch (err: any) {
            alert(err.response?.data?.message || 'Error al descargar examen');
        }
    };

    if (loading) {
        return (
            <LayoutPaciente>
                <div className="flex items-center justify-center h-64">
                    <Loader2 className="animate-spin h-12 w-12 text-blue-600" />
                </div>
            </LayoutPaciente>
        );
    }

    return (
        <LayoutPaciente>
            <div className="space-y-6">
                {/* Header */}
                <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
                    <div>
                        <h1 className="text-3xl font-bold text-gray-900 dark:text-white">Mis Exámenes</h1>
                        <p className="text-gray-600 dark:text-gray-400 mt-1">Consulta y descarga tus resultados médicos</p>
                    </div>

                    {/* Filtros */}
                    <div className="flex gap-2">
                        {[
                            { key: 'todos', label: 'Todos', color: 'blue' },
                            { key: 'pagados', label: 'Pagados', color: 'green' },
                            { key: 'pendientes', label: 'Pendientes', color: 'orange' },
                        ].map((item) => (
                            <button
                                key={item.key}
                                onClick={() => setFiltro(item.key)}
                                className={`px-4 py-2 rounded-lg font-medium transition ${filtro === item.key
                                        ? item.key === 'todos' ? 'bg-blue-600 text-white' :
                                            item.key === 'pagados' ? 'bg-green-600 text-white' :
                                                'bg-orange-600 text-white'
                                        : 'bg-white dark:bg-gray-800 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 border border-gray-200 dark:border-gray-700'
                                    }`}
                            >
                                {item.label}
                            </button>
                        ))}
                    </div>
                </div>

                {/* Error */}
                {error && (
                    <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 text-red-700 dark:text-red-400 px-4 py-3 rounded-lg">
                        {error}
                    </div>
                )}

                {/* Lista de Exámenes */}
                {examenes.length > 0 ? (
                    <div className="grid gap-6">
                        {examenes.map((examen) => (
                            <div key={examen.id} className="bg-white dark:bg-gray-800 rounded-xl shadow-md p-6 border border-gray-200 dark:border-gray-700 transition-colors">
                                <div className="flex flex-col lg:flex-row justify-between gap-4">
                                    {/* Info del Examen - SIEMPRE VISIBLE */}
                                    <div className="flex-1">
                                        <div className="flex items-start justify-between mb-3">
                                            <div>
                                                {/* NOMBRE DEL EXAMEN - SIEMPRE SE MUESTRA */}
                                                <h3 className="text-xl font-bold text-gray-900 dark:text-white">{examen.tipo_examen}</h3>
                                                <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">{examen.descripcion}</p>
                                            </div>
                                            <span className={`px-3 py-1 text-xs font-medium rounded-full ${examen.estado_pago === 'pagado'
                                                    ? 'bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-400'
                                                    : 'bg-orange-100 dark:bg-orange-900/30 text-orange-800 dark:text-orange-400'
                                                }`}>
                                                {examen.estado_pago_formateado}
                                            </span>
                                        </div>

                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                                            <div className="flex items-center">
                                                <Calendar className="w-4 h-4 mr-2 text-gray-400 dark:text-gray-500" />
                                                <span className="text-gray-500 dark:text-gray-400">Fecha:</span>
                                                <span className="ml-2 font-medium text-gray-900 dark:text-gray-200">{examen.fecha_formateada}</span>
                                            </div>
                                            <div className="flex items-center">
                                                <User className="w-4 h-4 mr-2 text-gray-400 dark:text-gray-500" />
                                                <span className="text-gray-500 dark:text-gray-400">Médico:</span>
                                                <span className="ml-2 font-medium text-gray-900 dark:text-gray-200">{examen.medico_solicitante}</span>
                                            </div>
                                            {examen.laboratorio_centro && (
                                                <div className="flex items-center">
                                                    <Building2 className="w-4 h-4 mr-2 text-gray-400 dark:text-gray-500" />
                                                    <span className="text-gray-500 dark:text-gray-400">Laboratorio:</span>
                                                    <span className="ml-2 font-medium text-gray-900 dark:text-gray-200">{examen.laboratorio_centro}</span>
                                                </div>
                                            )}
                                            {examen.estado_pago === 'pagado' && examen.metodo_pago && (
                                                <div className="flex items-center">
                                                    <CreditCard className="w-4 h-4 mr-2 text-gray-400 dark:text-gray-500" />
                                                    <span className="text-gray-500 dark:text-gray-400">Método Pago:</span>
                                                    <span className="ml-2 font-medium text-gray-900 dark:text-gray-200">{examen.metodo_pago}</span>
                                                </div>
                                            )}
                                        </div>

                                        {/* Monto - SIEMPRE VISIBLE */}
                                        {examen.monto > 0 && (
                                            <div className="mt-3 pt-3 border-t border-gray-200 dark:border-gray-700">
                                                <div className="flex items-center">
                                                    <DollarSign className="w-5 h-5 mr-1 text-blue-600 dark:text-blue-400" />
                                                    <span className="text-gray-700 dark:text-gray-300">Monto:</span>
                                                    <span className="text-lg font-bold ml-2 text-blue-600 dark:text-blue-400">${examen.monto.toFixed(2)}</span>
                                                    {examen.estado_pago === 'pagado' && examen.fecha_pago && (
                                                        <span className="ml-4 text-sm text-gray-600 dark:text-gray-400">
                                                            Pagado el {examen.fecha_pago}
                                                        </span>
                                                    )}
                                                </div>
                                            </div>
                                        )}
                                    </div>

                                    {/* Acciones - SOLO SI ESTÁ PAGADO */}
                                    <div className="flex flex-col justify-center gap-2 lg:w-48">
                                        {examen.puede_ver_resultado ? (
                                            <>
                                                <button
                                                    onClick={() => abrirVisorPDF(examen)}
                                                    className="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition font-medium flex items-center justify-center gap-2 shadow-sm"
                                                >
                                                    <Eye className="w-5 h-5" />
                                                    Ver
                                                </button>
                                                <button
                                                    onClick={() => descargarExamen(examen.id, examen.tipo_examen)}
                                                    className="w-full px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition font-medium flex items-center justify-center gap-2 shadow-sm"
                                                >
                                                    <Download className="w-5 h-5" />
                                                    Descargar
                                                </button>
                                            </>
                                        ) : examen.estado_pago === 'pendiente' ? (
                                            <div className="text-center p-4 bg-orange-50 dark:bg-orange-900/20 rounded-lg border border-orange-200 dark:border-orange-800">
                                                <p className="text-sm font-medium text-orange-800 dark:text-orange-400">⏳ Pendiente de Pago</p>
                                                <p className="text-xs text-orange-600 dark:text-orange-300 mt-1">Realiza el pago para acceder al resultado</p>
                                                <p className="text-lg font-bold text-orange-700 dark:text-orange-400 mt-2">
                                                    ${examen.monto.toFixed(2)}
                                                </p>
                                            </div>
                                        ) : (
                                            <div className="text-center p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-800">
                                                <p className="text-sm font-medium text-blue-800 dark:text-blue-400">⏳ Resultado Pendiente</p>
                                                <p className="text-xs text-blue-600 dark:text-blue-300 mt-1">El resultado será cargado próximamente</p>
                                            </div>
                                        )}
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                ) : (
                    <div className="bg-white dark:bg-gray-800 rounded-xl shadow-md p-12 text-center border border-gray-200 dark:border-gray-700 transition-colors">
                        <FileText className="w-16 h-16 mx-auto mb-4 text-gray-400 dark:text-gray-500" />
                        <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">No hay exámenes</h3>
                        <p className="text-gray-600 dark:text-gray-400">
                            {filtro === 'todos'
                                ? 'No tienes exámenes registrados aún'
                                : filtro === 'pagados'
                                    ? 'No tienes exámenes pagados'
                                    : 'No tienes exámenes pendientes de pago'
                            }
                        </p>
                    </div>
                )}

                {/* Modal del Visor PDF */}
                {pdfModalOpen && (
                    <div className="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80 backdrop-blur-sm">
                        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-2xl w-full max-w-6xl h-[90vh] flex flex-col border border-gray-200 dark:border-gray-700">
                            {/* Header del Modal */}
                            <div className="flex items-center justify-between p-4 border-b border-gray-200 dark:border-gray-700">
                                <div>
                                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                        {selectedExamen?.tipo_examen}
                                    </h3>
                                    <p className="text-sm text-gray-500 dark:text-gray-400">
                                        {selectedExamen?.fecha_formateada}
                                    </p>
                                </div>
                                <div className="flex items-center gap-2">
                                    <button
                                        onClick={() => selectedExamen && descargarExamen(selectedExamen.id, selectedExamen.tipo_examen)}
                                        className="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition font-medium flex items-center gap-2"
                                    >
                                        <Download className="w-4 h-4" />
                                        Descargar
                                    </button>
                                    <button
                                        onClick={cerrarVisorPDF}
                                        className="p-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-600 dark:text-gray-300"
                                    >
                                        <X className="w-6 h-6" />
                                    </button>
                                </div>
                            </div>

                            {/* Contenido del PDF */}
                            <div className="flex-1 p-4 overflow-hidden bg-gray-100 dark:bg-gray-900">
                                {pdfLoading ? (
                                    <div className="flex items-center justify-center h-full">
                                        <Loader2 className="animate-spin h-12 w-12 text-blue-600" />
                                    </div>
                                ) : pdfUrl ? (
                                    <iframe
                                        src={pdfUrl}
                                        className="w-full h-full rounded-lg border-0 bg-white"
                                        title="Visor de PDF"
                                    />
                                ) : (
                                    <div className="flex items-center justify-center h-full">
                                        <p className="text-gray-600 dark:text-gray-400">Error al cargar el PDF</p>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                )}
            </div>
        </LayoutPaciente>
    );
};

export default MisExamenes;
