import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import LayoutPaciente from '../../components/portal-paciente/LayoutPaciente';
import { User, Phone, Mail, Calendar, CreditCard, FileText, MapPin, Loader2 } from 'lucide-react';

interface PacienteData {
    id: number;
    ci: string;
    nombre: string;
    apellido: string;
    nombre_completo: string;
    celular: string;
    email: string | null;
    sexo: string;
    fecha_nacimiento: string | null;
    direccion: string | null;
    numero_historia: string;
    estado: string;
    created_at: string;
}

const PerfilPaciente: React.FC = () => {
    const navigate = useNavigate();
    const [paciente, setPaciente] = useState<PacienteData | null>(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState('');

    useEffect(() => {
        cargarPerfil();
    }, []);

    const cargarPerfil = async () => {
        try {
            const token = localStorage.getItem('paciente_token');
            const pacienteId = localStorage.getItem('paciente_id');

            if (!token || !pacienteId) {
                navigate('/paciente/login');
                return;
            }

            const response = await axios.get('/paciente/me', {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'X-Paciente-Id': pacienteId
                }
            });

            if (response.data.success) {
                setPaciente(response.data.data);
            }
        } catch (err: any) {
            console.error('Error al cargar perfil:', err);
            if (err.response?.status === 401) {
                localStorage.removeItem('paciente_token');
                localStorage.removeItem('paciente');
                localStorage.removeItem('paciente_id');
                navigate('/paciente/login');
            } else {
                setError('Error al cargar tu perfil');
            }
        } finally {
            setLoading(false);
        }
    };

    const formatDate = (dateString: string | null) => {
        if (!dateString) return 'No registrado';
        return new Date(dateString).toLocaleDateString('es-ES', {
            day: '2-digit',
            month: 'long',
            year: 'numeric'
        });
    };

    if (loading) {
        return (
            <LayoutPaciente>
                <div className="flex items-center justify-center h-64">
                    <Loader2 className="animate-spin h-12 w-12 text-blue-600" />
                </div>
            </LayoutPaciente>
        );
    }

    if (error || !paciente) {
        return (
            <LayoutPaciente>
                <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
                    {error || 'Error al cargar datos'}
                </div>
            </LayoutPaciente>
        );
    }

    return (
        <LayoutPaciente>
            <div className="space-y-6">
                {/* Título */}
                <div>
                    <h1 className="text-3xl font-bold text-gray-900 dark:text-white">Mi Perfil</h1>
                    <p className="text-gray-600 dark:text-gray-400 mt-1">Información personal registrada</p>
                </div>

                {/* Avatar y Nombre */}
                <div className="bg-white dark:bg-gray-800 rounded-xl shadow-md p-6 border border-gray-200 dark:border-gray-700 transition-colors">
                    <div className="flex flex-col md:flex-row items-center md:items-start gap-6">
                        <div className="w-24 h-24 bg-gradient-to-br from-blue-600 to-cyan-600 rounded-full flex items-center justify-center text-white text-3xl font-bold shadow-lg">
                            {paciente.nombre.charAt(0)}{paciente.apellido.charAt(0)}
                        </div>
                        <div className="text-center md:text-left">
                            <h2 className="text-2xl font-bold text-gray-900 dark:text-white">{paciente.nombre_completo}</h2>
                            <p className="text-gray-600 dark:text-gray-400">Paciente</p>
                            <div className="mt-2 flex flex-wrap gap-2 justify-center md:justify-start">
                                <span className="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-400">
                                    Activo
                                </span>
                                <span className="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-400">
                                    Historia: {paciente.numero_historia}
                                </span>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Datos Personales */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    {/* Información de Identidad */}
                    <div className="bg-white dark:bg-gray-800 rounded-xl shadow-md p-6 border border-gray-200 dark:border-gray-700 transition-colors">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4 flex items-center">
                            <CreditCard className="w-5 h-5 mr-2 text-blue-600 dark:text-blue-400" />
                            Información de Identidad
                        </h3>
                        <div className="space-y-4">
                            <div className="flex items-start">
                                <div className="flex-shrink-0">
                                    <FileText className="w-5 h-5 text-gray-400 dark:text-gray-500 mt-0.5" />
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Cédula de Identidad</p>
                                    <p className="text-base text-gray-900 dark:text-gray-100">{paciente.ci}</p>
                                </div>
                            </div>
                            <div className="flex items-start">
                                <div className="flex-shrink-0">
                                    <User className="w-5 h-5 text-gray-400 dark:text-gray-500 mt-0.5" />
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Nombre Completo</p>
                                    <p className="text-base text-gray-900 dark:text-gray-100">{paciente.nombre_completo}</p>
                                </div>
                            </div>
                            <div className="flex items-start">
                                <div className="flex-shrink-0">
                                    <User className="w-5 h-5 text-gray-400 dark:text-gray-500 mt-0.5" />
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Sexo</p>
                                    <p className="text-base text-gray-900 dark:text-gray-100">{paciente.sexo}</p>
                                </div>
                            </div>
                            <div className="flex items-start">
                                <div className="flex-shrink-0">
                                    <Calendar className="w-5 h-5 text-gray-400 dark:text-gray-500 mt-0.5" />
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Fecha de Nacimiento</p>
                                    <p className="text-base text-gray-900 dark:text-gray-100">{formatDate(paciente.fecha_nacimiento)}</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Información de Contacto */}
                    <div className="bg-white dark:bg-gray-800 rounded-xl shadow-md p-6 border border-gray-200 dark:border-gray-700 transition-colors">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4 flex items-center">
                            <Phone className="w-5 h-5 mr-2 text-blue-600 dark:text-blue-400" />
                            Información de Contacto
                        </h3>
                        <div className="space-y-4">
                            <div className="flex items-start">
                                <div className="flex-shrink-0">
                                    <Phone className="w-5 h-5 text-gray-400 dark:text-gray-500 mt-0.5" />
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Celular</p>
                                    <p className="text-base text-gray-900 dark:text-gray-100">{paciente.celular}</p>
                                </div>
                            </div>
                            <div className="flex items-start">
                                <div className="flex-shrink-0">
                                    <Mail className="w-5 h-5 text-gray-400 dark:text-gray-500 mt-0.5" />
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Correo Electrónico</p>
                                    <p className="text-base text-gray-900 dark:text-gray-100">{paciente.email || 'No registrado'}</p>
                                </div>
                            </div>
                            <div className="flex items-start">
                                <div className="flex-shrink-0">
                                    <MapPin className="w-5 h-5 text-gray-400 dark:text-gray-500 mt-0.5" />
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Dirección</p>
                                    <p className="text-base text-gray-900 dark:text-gray-100">{paciente.direccion || 'No registrada'}</p>
                                </div>
                            </div>
                            <div className="flex items-start">
                                <div className="flex-shrink-0">
                                    <Calendar className="w-5 h-5 text-gray-400 dark:text-gray-500 mt-0.5" />
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Fecha de Registro</p>
                                    <p className="text-base text-gray-900 dark:text-gray-100">{formatDate(paciente.created_at)}</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Nota Informativa */}
                <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-xl p-4">
                    <div className="flex items-start">
                        <div className="flex-shrink-0">
                            <FileText className="h-5 w-5 text-blue-600 dark:text-blue-400" />
                        </div>
                        <div className="ml-3">
                            <h3 className="text-sm font-medium text-blue-800 dark:text-blue-400">Información importante</h3>
                            <div className="mt-2 text-sm text-blue-700 dark:text-blue-300">
                                <p>Los datos de tu perfil son gestionados por el personal médico.</p>
                                <p className="mt-1">Si necesitas actualizar alguna información, contacta a tu médico o a recepción.</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </LayoutPaciente>
    );
};

export default PerfilPaciente;
