<?php

use App\Http\Controllers\API\AuthController;
use App\Http\Controllers\API\CitaController;
use App\Http\Controllers\API\PacienteController;
use App\Http\Controllers\API\ResultadoController;
use App\Http\Controllers\API\UsuarioController;
use App\Http\Controllers\API\EstudioController;
use App\Http\Controllers\API\CategoriaController;
use App\Http\Controllers\API\ExamenController;
use App\Http\Controllers\API\PagoExamenController;
use App\Http\Controllers\Portal\PacienteAuthController;
use App\Http\Controllers\Portal\PacienteDashboardController;
use App\Http\Controllers\Portal\PacienteExamenController;
use App\Http\Controllers\Portal\PacienteHistoriaController;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes - Cibertronia Salud
|--------------------------------------------------------------------------
*/

// ========================================
// RUTAS PÚBLICAS (sin autenticación)
// ========================================
Route::post('/login', [AuthController::class, 'login']);

// Login del paciente (portal)
Route::post('/pacientes/login', [PacienteController::class, 'loginPaciente']);

// Rutas públicas para descargar/ver resultados (requieren que el paciente haya pagado)
Route::get('resultados/{id}/download', [ResultadoController::class, 'download']);
Route::get('resultados/{id}/view', [ResultadoController::class, 'view']);

// SOLUCIÓN 3: Endpoint público para estudios (lectura solamente)
Route::get('/estudios-publicos', [EstudioController::class, 'indexPublico']);

// Ruta de prueba (sin autenticación)
Route::get('/health', function () {
    return response()->json([
        'success' => true,
        'message' => 'Cibertronia Salud API está funcionando correctamente',
        'timestamp' => now()->toDateTimeString(),
    ]);
});

// ========================================
// PORTAL DEL PACIENTE
// ========================================
Route::prefix('paciente')->group(function () {
    // Login del paciente
    Route::post('/login', [PacienteAuthController::class, 'login']);
    
    // Rutas protegidas del paciente
    Route::middleware('auth.paciente')->group(function () {
        Route::post('/logout', [PacienteAuthController::class, 'logout']);
        Route::get('/me', [PacienteAuthController::class, 'me']);
        Route::put('/cambiar-password', [PacienteAuthController::class, 'cambiarPassword']);
        
        // Dashboard
        Route::get('/dashboard', [PacienteDashboardController::class, 'index']);
        
        // Exámenes del paciente
        Route::get('/examenes', [PacienteExamenController::class, 'index']);
        Route::get('/examenes/{id}', [PacienteExamenController::class, 'show']);
        Route::get('/examenes/{id}/descargar', [PacienteExamenController::class, 'descargar']);
        Route::get('/examenes/{id}/ver', [PacienteExamenController::class, 'ver']);

        // Historial clínico del paciente
        Route::get('/historias', [PacienteHistoriaController::class, 'index']);
    });
});

// ========================================
// RUTAS PROTEGIDAS - ADMIN
// ========================================
// Usamos nuestro middleware personalizado
Route::middleware([\App\Http\Middleware\ForceSanctumAuth::class])->group(function () {
    
    // **Autenticación**
    Route::post('/logout', [AuthController::class, 'logout']);
    Route::get('/user', [AuthController::class, 'user']);
    Route::put('/user/profile', [AuthController::class, 'updateProfile']);
    Route::put('/user/password', [AuthController::class, 'changePassword']);

    // **Usuarios**
    Route::apiResource('usuarios', UsuarioController::class);
    Route::patch('usuarios/{id}/password', [UsuarioController::class, 'cambiarPassword']);

    // **Categorías**
    Route::apiResource('categorias', CategoriaController::class);
    Route::get('categorias/activas', [CategoriaController::class, 'activas']);

    // **Estudios**
    Route::apiResource('estudios', EstudioController::class);
    Route::get('estudios/categorias', [EstudioController::class, 'categorias']);
    Route::get('estudios/categoria/{categoria}', [EstudioController::class, 'porCategoria']);

    // **Pacientes**
    Route::apiResource('pacientes', PacienteController::class);
    Route::get('pacientes/ci/{ci}', [PacienteController::class, 'buscarPorCi']);
    Route::patch('pacientes/{id}/cambiar-estado-pago', [PacienteController::class, 'cambiarEstadoPago']);

    // **Citas**
    Route::apiResource('citas', CitaController::class);
    Route::get('citas/hoy', [CitaController::class, 'hoy']);
    Route::get('citas/agenda/{medicoId}', [CitaController::class, 'agenda']);
    // Legacy: columna 'pagado' no existe en tabla citas — ruta neutralizada
    Route::patch('citas/{id}/pagado', function (\Illuminate\Http\Request $request, $id) {
        return response()->json(['success' => true, 'pagado' => false, 'message' => 'No soportado en versión actual']);
    });
    Route::post('citas/verificar-disponibilidad', [CitaController::class, 'verificarDisponibilidad']);

    // **Historia Clínica (Fase 2: SOAP + CIE-10 + Archivos)**
    Route::apiResource('historias-clinicas', \App\Http\Controllers\API\HistoriaClinicaController::class)
        ->middleware(\App\Http\Middleware\AuditoriaHistoriaClinica::class);
    Route::post('historias-clinicas/{id}/evoluciones', [\App\Http\Controllers\API\HistoriaClinicaController::class, 'agregarEvolucion']);
    Route::post('historias-clinicas/{id}/archivos', [\App\Http\Controllers\API\HistoriaClinicaController::class, 'subirArchivosAdjuntos']);
    Route::get('cie10/buscar', [\App\Http\Controllers\API\HistoriaClinicaController::class, 'buscarCie10']);

    // **Recetas Médicas**
    Route::apiResource('recetas', \App\Http\Controllers\API\RecetaMedicaController::class);
    Route::get('recetas/paciente/{pacienteId}', [\App\Http\Controllers\API\RecetaMedicaController::class, 'porPaciente']);

    // **Catálogo de Medicamentos**
    Route::get('medicamentos', [\App\Http\Controllers\API\RecetaMedicaController::class, 'catalogo']);
    Route::get('medicamentos/buscar', [\App\Http\Controllers\API\RecetaMedicaController::class, 'buscar']);
    Route::get('medicamentos/categoria/{categoria}', [\App\Http\Controllers\API\RecetaMedicaController::class, 'porCategoria']);
    Route::get('medicamentos/categorias', [\App\Http\Controllers\API\RecetaMedicaController::class, 'categorias']);

    // **Exámenes Médicos** (NUEVO)
    Route::apiResource('examenes', ExamenController::class);
    Route::post('examenes/{id}/subir-resultado', [ExamenController::class, 'subirResultado']);

    // **Pagos de Exámenes** (NUEVO)
    Route::apiResource('pagos-examenes', PagoExamenController::class);
    Route::post('examenes/{id}/pago', [PagoExamenController::class, 'registrarPago']);
    Route::patch('pagos-examenes/{id}/estado', [PagoExamenController::class, 'actualizarEstado']);

    // **Resultados**
    Route::apiResource('resultados', ResultadoController::class);
    Route::get('resultados/paciente/{pacienteId}', [ResultadoController::class, 'porPaciente']);

    // **Antecedentes (NUEVO)**
    Route::get('pacientes/{id}/antecedentes', function($id) {
        return response()->json(\Illuminate\Support\Facades\DB::table('antecedentes_medicos')->where('paciente_id', $id)->get());
    });

    // **Dashboard / Estadísticas**
    Route::prefix('dashboard')->group(function () {
        Route::get('/estadisticas', function () {
            try {
                return response()->json([
                    'success' => true,
                    'data' => [
                        'total_pacientes'      => \App\Models\Paciente::activos()->count(),
                        'pacientes_pendientes' => \App\Models\Paciente::pendientes()->count(),
                        'pacientes_pagados'    => \App\Models\Paciente::pagados()->count(),
                        'total_citas'          => \App\Models\Cita::count(),
                        'total_usuarios'       => \App\Models\Usuario::where('estado', 'activo')->count(),
                        'citas_hoy'            => \App\Models\Cita::whereDate('fecha_cita', today())->count(),
                    ],
                ]);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => true,
                    'data' => [
                        'total_pacientes'      => 0,
                        'pacientes_pendientes' => 0,
                        'pacientes_pagados'    => 0,
                        'total_citas'          => 0,
                        'total_usuarios'       => 0,
                        'citas_hoy'            => 0,
                        'error'                => $e->getMessage(),
                    ],
                ]);
            }
        });
    });
});
